function [age,b_dot,fcp,xcb,xcb0,ct1,ct2,ct3,ct4]=...
    XCM_simp_Nye_v2(T,X,W,acp,H,n_steps,nrep,varargin)
%This function takes N input signals and increases their
%cross-correlation via simulated annealing.  This is simplified
%version of XCMNgrid.m which also has features for evaluating
%whether the synchronization is unique but runs much more slowly.
%
%     Run Version 2: XCM_simp_Nye_v2 randombly perturbs the interval length
%     between floating control points while holding the interval b/w ACP's
%     constant. Perturbations follow log normal distribution and are 
%     systematically decreased with improved fit. Standard Deviation of 
%     floating control points follow Brownian Bridge b/w ACP's  
% 
%     (see XCM_simp, XCM_simp_Nye)
% 
%Written for Matlab 7.  
%
%function [age,b_dot,fcp,xcb,xcb0,GoF,GoF0,ct2,ct3,ct4]=...
%    XCM_simp_Nye_v2(T,X,W,acp,H,n_steps,nrep,varargin)
%
%Inputs 
% T:  target record(s), cell format w/ each timeseries in a different cell
%    T{n} = [t s]:  time-scale and signal for target record n
%
% X:  record(s) to be adjusted
%    X{n} = [d s]:   initial depth (ice equivalent) and signal
%
% W:  weights applied to the cross-correlation between T.s and X.s
%     *if there are multiple signals for T and X (i.e. d18O-O2, CH4, &
%     CO2), the goodness value for a chronology is the weighted value of
%     correlations of each signal.
%
% acp: age-control-points used in adjusting X{n}[d] to T{n}[t].  
%
% nrep: number of repetitions used in searching for a better match
%
% H: Ice sheet thickness (ice equivalent)
%
%Optional Inputs:
% fix_ends
%
% a_thresh
% a_max
% a_rate_change
%
% delta_age
% ref_date
% date0
%
%Output:
% age - Best fit ages for X (will be same size as X)
% b_dot - FCP & Best fit accumulation b/w FCP
% xcb - GoF for best iteration (Chi^2 | R^2 | StDev)
% xcb0 - Prior xcb
% GoF - GoF (2) (Deviance: Improvement of probability of Chi^2 given n)
% GoF0 - Prior GoF
% ct1 - Number of iterations
% ct2 - Number of iterations that meet qualifications
% ct3 - Number of iterations that improve fit
% ct4 - 
% 

%% Formatting input fields
if rem(size(varargin,2),2)==0
    fields=reshape(varargin,2,size(varargin,2)/2);
    fields=cell2struct(fields(2,:),fields(1,:),2);
else
    fprintf('Input fields must enter as pairs')
end


%% Time Series formatting
if isnumeric(X); X={X}; x_cell=1; else x_cell=0; end
if isnumeric(T); T={T}; t_cell=1; else t_cell=0; end

if isstruct(X) || isstruct(T); error('X | T should be numeric or cell inputs (see function description)'); end
%can put in text here to restructure T & X to cell

if ~isvector(T);error('Error in Reference Format: Improper size');end
if ~isvector(X);error('Error in New Timeseries Format: Improper size');end

nrec=numel(X); %number of records to be adjusted

if ~isequal(numel(X),numel(T)); error('Error: Reference Timeseries Input and New Timeseries have different format sizes');end

%Check formating of Reference Timeseries records
for rec=1:nrec;
    [Ndepths,Nmeas]=size(T{rec});
    if Nmeas<2;error('Error in Reference Format: Must include depth and measurements arranged in columns\n');end
    if Ndepths<2;error('Error in Reference Format: Must include timeseries\n');end
    if ~issorted(T{rec}(:,1));error('Error in Reference Format: Depths not sorted\n');end
    if sum(isnan(T{rec}(:,2)));fprintf('Reference timeseries signal #%.0f contain NaN values: NaN values eliminated\n',rec);
        i=isnan(T{rec}(:,2));T{rec}=T{rec}(~i,:);end
    if isequal(0,std(T{rec}(:,2)));error('Error in Reference Format: Reference timeseries is constant\n');end

    [Mdepths,Mmeas]=size(X{rec});
    if Mmeas<2;error('Error in New Timeseries Format: Must include depth and measurements arranged in columns\n');end
    if Mdepths<2;error('Error in New Timeseries Format: Must include timeseries\n');end
    if ~issorted(X{rec}(:,1));error('Error in New Timeseries Format: Depths not sorted\n');end
    if sum(isnan(X{rec}(:,2)));fprintf('New timeseries signal #%.0f contain NaN values: NaN values eliminated\n',rec);
        i=isnan(X{rec}(:,2));X{rec}=X{rec}(~i,:);end
    if isequal(0,std(X{rec}(:,2)));error('Error in New Timeseries Format: Reference timeseries is constant\n');end
end

if isempty(W); W=ones(nrec,1);end
if ~isequal(numel(W),nrec);error('Error in weighting of signals');end

%% Check Age Control Points
if sum(sum(isnan(acp)));error('Error in Age Control Points: NaN values found');end


%% Default Parameters
fix_ends='both';
a_thresh=3;
a_max=1;
a_rate_change=inf;

if isfield(fields,'fix_ends'); fix_ends=fields.fix_ends; end
if isfield(fields,'a_thresh'); a_thresh=fields.a_thresh; end
if isfield(fields,'a_max'); a_max=fields.a_max; end
if isfield(fields,'a_rate_change'); a_rate_change=fields.a_rate_change; end


%% Default Nye Parameters (really these need to be included as input)
delta_age=0;
ref_date=1950;
date0=2012;
if isfield(fields,'delta_age');delta_age=fields.delta_age;end
if isfield(fields,'ref_date');ref_date=fields.ref_date;end
if isfield(fields,'date0');date0=fields.date0;end


%% Normalize series to std=1, mean = 5;
sig=1;
mu=5;
for rec=1:nrec
    age0{rec}=Nye_interp(X{rec}(:,1),acp,H,...
        'delta_age',delta_age,'ref_date',ref_date,'date0',date0);
    
    %Geometric mean and std of X timeseries
    X_mean=trapz(age0{rec},X{rec}(:,2))/range(age0{rec});
    X_std=sqrt(sum((X{rec}(:,2)-X_mean).^2)/(numel(X{rec}(:,2))-1));
    %Normalization of X and T timeseries
    X{rec}(:,2)=sig*(X{rec}(:,2)-X_mean)/X_std+mu;
    T{rec}(:,2)=sig*(T{rec}(:,2)-X_mean)/X_std+mu;
end

%%
% Break record into Nsteps
X_min=min(cellfun(@min,age0));
X_max=max(cellfun(@max,age0));

%define floating control points
%     -This method does not allow for extrapolation of data older/younger than the
%     oldest/youngest ACP

%Additional FCP to complete Brownian Bridge to next acp
dk_i=(X_max-X_min)/n_steps;
i_acpi=find(acp(:,2)<X_min,1,'last');
i_acpf=find(acp(:,2)>X_max,1,'first');
FCP=unique([acp(i_acpi:i_acpf,2);X_min;X_max]);

dk_fcp=round(diff(FCP)/dk_i);
dk_fcp(dk_fcp==0)=1;

%Divide into subsections roughly 'dk_i' between acp
fcp=[];
for i=1:numel(dk_fcp);
    fcp=[fcp;linspace(FCP(i),FCP(i+1),dk_fcp(i)+1)'];
end
fcp=unique(fcp);

%interpolate the depths for these ages
fcp(:,2)=fcp;
fcp(:,1)=NyeInv_interp(fcp(:,2),acp,H,...
    'delta_age',delta_age,'ref_date',ref_date,'date0',date0);

[~,i_acp]=intersect(fcp(:,2),acp(:,2));


%% Perturbation Parameters
i_fcp=fcp(:,2)>=X_min & fcp(:,2)<=X_max;
% dk=mean(diff(fcp(i_fcp,2))); % mean difference in age control points
% k=linspace(1/100,a_rate_change/10,10); %Perturbation size (linearly decreasing, set # of refinements)
k=logspace(log10(1/128),log10(a_rate_change/8),13); %Perturbation size (logarithmically decreasing)
mu=log(1./sqrt(k.^2+1)); % parameters for lognormal perturbations
sigma=sqrt(log(k.^2+1)); % parameters for lognormal perturbations

%% Initial cross-correlation (xc1)
xc1=nan(1,nrec);
xc2=nan(1,nrec);
xc1_stat=nan(1,nrec);
xc2_stat=nan(1,nrec);

for rec=1:nrec
    
    pl1{rec}=~isnan(X{rec}(:,2));
    pl2{rec}=~isnan(T{rec}(:,2));
    
    %interpolate T signal at ages of X
    tsi=interp1(T{rec}(pl2{rec},1),T{rec}(pl2{rec},2),age0{rec}(pl1{rec})); 
    %GoF 1: Pearson Cross-Correlation
%     pl=~isnan(tsi);
%     xc2(rec)=xcPH(X(rec).s(pl),tsi(pl),0); %cross correlation of X w/ Ref
    %GoF 2: Chi^2 
    [~,xc2(rec),xc2_stat(rec)]=chi2test2(tsi,X{rec}(pl1{rec},2)); %Pearson Chi^2 \
    %GoF 3: Standard Deviation
%     xc2(rec)=sum((tsi-X{rec}(pl1{rec},2)).^2)/sum(~isnan(tsi));   
%    GoF 4: Deviance (Probability of Chi^2 value given N population)
%     GoF0(:,rec)=[chi2cdf(xc1(rec),numel(xsi));chi2cdf(xc2(rec),numel(tsi))];
%     GoF0(:,rec)=[chi2cdf(xc2(rec),numel(tsi))];

end

% xcb=[xc1;xc2];
xcb=xc2;
xcb0=xcb;

% Weighted sum of chi^2 values
xcb_sum=xcb*W';
xcb_sum0=xcb_sum;

%% Parameters for Synchronization loop
fcp0=fcp(:,2);
% # of control points
ncp=numel(fcp(:,2)); 

% Counters
ct=0; % counter
ct1=0; %Counts all perturbations
ctt=0;
ct2=0; %counts number of iterations that meet thresholds (a_thres, a_max)
ct3=0; %counts number of new "best" acp iterations;

% # of refinement steps
kpl=length(k);

% GoF=nan;

% ct4=[0 0 0 0];%Counter used for development
ct4=zeros(1,kpl);
% ctt2=0;

% Can age of first and/or last data point of X be adjusted?
switch fix_ends
    case 'both'
        nn=[0;0];
    case 'front'
        nn=[0;1];
    case 'back'
        nn=[1;0];
    case 'none'
        nn=[1;1];
end

% length of interval b/w initial fcp's
dfcp0=diff(fcp0(i_fcp));

%% Synchronization loop
while kpl>0
    %---Perturbation step----%
    ct1=ct1+1;
    % ctt2=ctt2+1;
    
    %Perturb dfcp
    dfcp=diff(fcp(:,2));
    dfcp2=dfcp.*(lognrnd(mu(kpl),sigma(kpl),[ncp-1,1]));

    %Scale the length of dfcp2 to remain consistent with acp's
        %i_acp = vector of positions of acp's in fcp;
    fcp2=[fcp(1,2);fcp(1,2)+cumsum(dfcp2)];
    fcp2=interp1(fcp2(i_acp),fcp(i_acp,2),fcp2);
    
    %New dfcp2
    dfcp2=diff(fcp2(i_fcp));
    
    %---Check against criteria if this is a legal perturbation -----%
    %Did the random adjustment to FCP incr/decr accumulation rate by factor of 2 (a_thresh) from original FCPs?
    cri1=sum(dfcp2./dfcp0<1/a_thresh | dfcp2./dfcp0>a_thresh);  
    if ~cri1 %&& ...
        % Check Rate of Change of "accumulation" rate
        btt=H*log((H-fcp(1:end-1,1))./(H-fcp(2:end,1)))./diff(fcp2);
        btt_dot=[max(btt(2:end)./btt(1:end-1));min(btt(2:end)./btt(1:end-1))];
        
        cri2=max(btt)<a_max;
        cri3=btt_dot(1)<a_rate_change;
        cri4=btt_dot(2)>1/a_rate_change;
        
        % ct4=ct4+[ctt2-1 ~cri2 ~cri3 ~cri4];
        % ctt2=0;
        % ct4=ct4+1; %number of perturbations passing monopl criteria
        if cri2 && ...
               cri3  && ...
                cri4;
            
            ct=ct+1; 
            
        %---Calculate new GoF ----%
            ct2=ct2+1; %number of iterations passing all criteria
            for rec=1:nrec
                % Translate best guess X-ages to new FCP2 from original FCP
                xtt{rec}=Nye_interp(X{rec}(:,1),[fcp(:,1),fcp2],H,...
                    'delta_age',delta_age,'ref_date',ref_date,'date0',date0);    
                % Interpolate Ref signal at new X-ages
                tsi=interp1(T{rec}(pl2{rec},1),T{rec}(pl2{rec},2),xtt{rec}(pl1{rec})); 
                [~,xc2(rec),xc2_stat(rec)]=chi2test2(tsi,X{rec}(pl1{rec},2)); %Pearson Chi-2 

            end
            xcb_sum_i=xc2*W';

            
        %---If GoF improves --> Make new base to iterate from -----%
            if xcb_sum_i(1)<=xcb_sum(1)% && xcb_sum_i(2)<xcb_sum(2)  %xc1(end)<xcb && xc2(end)<xcb
                age=xtt;
                b_dot=btt;
                % xcb=min([xc1(nrec+1) xc2(nrec+1)]);
                % xcb=max([xc1(end) xc2(end)]);
                % xcb=[xc1;xc2];
                xcb=[xc2];
                xcb_sum=xcb_sum_i;
                % GoF=GoFi;
                fcp(:,2)=fcp2; 
                ct3=ct3+1; %number of successful interations
                ct=0; 
                ct4(kpl)=ct4(kpl)+1;
            end
            ctt=ct1;
        end
    end

    if ct>=nrep || ct1-ctt>50000; kpl=kpl-1; ct=0; end;
end


%% Define Output

if ct3;
    b_dot=[fcp(i_fcp,:) b_dot(i_fcp)];
    fcp=fcp(i_fcp,:);
else
    age=age0;
    b_dot=nan(size(fcp,1),3);
    fprintf('FCP not improved?\n\n')
end

% if x_cell; age=age{1}; b_dot=b_dot{1}; end