%% Synchronization Parameters
MIS=1; %choose either 1 (Holocene) or 2 (30 ka - 11.7 ka)

%Monte Carlo parameters
W=[1 1]; %Weighting of timeseries (CH4 and d18Oatm)

N=1000; % Number of MC realizations
nrep=20; % Number of iterations in synchronization routine before refining perturbations

interp='Nye'; % How to interpolate b/w age constraints ('Nye','Linear','DJ')
X_err='no_error'; % Include error in RICE timeseries (CH4 and d18Oatm)
T_err='no_error'; % Include error in WDC timeseries

if MIS==1
    fix_ends='none'; %Freeze age of either/both/neither of the first & last data point
    nsteps=75; %75 for Holocene (0-11750)
elseif MIS==2
    fix_ends='front'; %For MIS2 hold first data point fixed (this was determined from MIS=1)
    nsteps=25; %25 for MIS2 (11750-30000)
end

a_thresh=5; %Maximum change in ALT (iteration/priori)
a_max=.75; %Maximum allowable accumulation rate
a_rate_change=2; %Maximum ratio of change in accumulation between subsections


% Filtering Output
% 1) Correlation > R2_thresh
% 2) Best % of iterations
R2_thresh=inf; % Cutoff for minimal GoF for realization to be considered
pt=1; %Percentile cutoff


%%
load('MC_Synchronization_Example_Data')

%% Monte Carlo Parameters
% Setup Reference timeseries for MC (WDC CH4 and d18Oatm)
T0=[{WDC.CH4} {WDC.d18O2}];
for rec=1:nrec;
    %prior age of RICE
    age0{rec}=age0{rec}(~isnan(X0{rec}(:,2)),:);
    %remove NaN values from RICE records
    X0{rec}=X0{rec}(~isnan(X0{rec}(:,2)),:);
    %remove NaN values from WDC records
    T0{rec}=T0{rec}(~isnan(T0{rec}(:,2)),:);
    
    %average replicates samples in RICE
    [~,X0{rec}]=NUV(X0{rec},1,'occurrence','mean');
    %average replicates samples in WDC
    [~,T0{rec}]=NUV(T0{rec},1,'occurrence','mean');
    
    %RICE uncertainty values: if NaN --> mean uncertainty
    X0{rec}(isnan(X0{rec}(:,3)),3)=nanmean(X0{rec}(:,3));
    %WDC uncertainty values: if NaN --> mean uncertainty
    T0{rec}(isnan(T0{rec}(:,3)),3)=nanmean(T0{rec}(:,3));
    T0{rec}(isnan(T0{rec}(:,3)),3)=nanmean(X0{rec}(:,3));
end
X=X0;
T=T0;


%% Pre-Define some of the output variables;
% Perturb prior acp's
acp2=nan([size(acp0),N]);
acp2(:,1,:)=reshape(acp0(:,1)*ones(1,N),[size(acp0,1),1,N]);
acp2=reshape(acp0(:,2)*ones(1,N)+randn(size(acp0,1),N).*(acp0(:,3)*ones(1,N)),[size(acp0,1),1,N]);

chi2_i=nan(1,nrec,N); % GoF
chi2_0_i=nan(1,nrec,N); % GoF of before synchronization
% GoF0=nan(1,nrec,N);
% GoF=nan(1,nrec,N);

ct1=nan(N,1); %Number of iterations during synchronization
ct2=nan(N,1); %Number of iterations that meet qualifiers
ct3=nan(N,1); %Number of iterations that improve fit
ct4=nan(N,1); %Developmental counter for trouble shooting

b_dot=nan(nsteps+2,3,N); %"Accumulation" from Nye interpretation

for rec=1:nrec
    [n,m]=size(T{rec}(:,1:2));
    in1{rec}=nan(n,2,N); % Record WDC input to synchronization
    
    [n,m]=size(X{rec}(:,1:2));
    in2{rec}=nan(n,2,N); % Record RICE input to synchronization)
    
    out{rec}=nan(n,2,N); % Output from synchronization (RICE)
end


%% Run MC Loop
tic
for i=1:N
    if strcmp(T_err,'error')
        for rec=1:nrec
            T{rec}(:,2)=T0{rec}(:,2)+randn(size(T0{rec}(:,2))).*T0{rec}(:,3);
            in1{rec}(:,:,i)=T{rec}(:,1:2);
        end
    end
    if strcmp(X_err,'error')
        for rec=1:nrec
            X{rec}(:,2)=X0{rec}(:,2)+randn(size(X0{rec}(:,2))).*X0{rec}(:,3);
            in2{rec}(:,:,i)=X{rec}(:,1:2);
        end
    end
    
    [out_i,~,fcp{i},chi2_i(:,:,i),chi2_0_i(:,:,i),ct1(i),ct2(i),ct3(i)]=...
        XCM_simp_Nye_v2(T,X,W,acp2(:,:,i),H_ice_eq,nsteps,nrep,...
        'fix_ends',fix_ends,'a_thresh',a_thresh,'a_max',a_max,'a_rate_change',a_rate_change,...
        'delta_age',delta_age,'ref_date',ref_date,'date0',date0);

    % Correlation Routine Stats
    fprintf('Annealing Run # %.0f: Perturbations Attempts: %.0f, Passing Pert''s: %.0f, Successful iterations: %.0f\n',i,ct1(i),ct2(i),ct3(i))
    fprintf('Initial R2 = %.4f & %.4f --> final R2 = %.4f & %.4f \n',chi2_0_i(:,:,i)',chi2_i(:,:,i)')
    if ~rem(i,10)
        disp(['i = ' num2str(i) '  :  run time = ' num2str(toc)])
    end
    
    %Assemble output
    for rec=1:nrec
        out{rec}(:,:,i)=[out_i{rec} X{rec}(:,2)];
    end
    
end

for i=1:i; chi2_c(:,:,i)=chi2_i(:,:,i)*W';end

% Save .mat file of Monte Carlo results
date=datevec(now);
save_file_name=sprintf('MC3_%.0f%2.0f%.0f_MIS%.0f_%s_N%.0f_nsteps%.0f_nrep%.0f_sm100_Disc',date(1:3),MIS,X_err,N,nsteps,nrep);
save_file_name=strrep(save_file_name,' ','0')
save(['C:\Users\James Lee\Google Drive\Matlab\RICE\Gas Match\' save_file_name],...
    'MIS','ACP','acp0','acp2','RICE','WDC','age0','X','T','nrep','nsteps','N',...
    'fix_ends','a_thresh','a_max','a_rate_change','delta_age','ref_date','date0',...
    'fcp','chi2_i','chi2_0_i','out','W','chi2_c','ct1','ct2','ct3')%,...'b_dot','ct4','in1','in2'
    

%% Sorting Output for Plots
% Sort realizations from worst to best GoF
[chi2_c2,i2]=sort(chi2_c(1,1,:),3,'descend');
fcp2=fcp(i2);
for rec=1:nrec
    out2{rec}=out{rec}(:,:,i2);
end

% Find iterations better than threshold
id=chi2_c2<=R2_thresh;
% Find top 'pt' percent of those iterations
i_pct=floor(N*(1-pt));
if i_pct;id(1:i_pct)=0;end


%% Calculate error about best chronology

%interpolate age of depths from fcp_best by age constraints of all
%realizations
fcp2_interp_age=nan(numel(fcp2{end}(:,1)),N);
for i=1:N;
    fcp2_interp_age(:,i)=Nye_interp(fcp2{end}(:,1),fcp{i},H_ice_eq,'delta_age',delta_age,'ref_date',ref_date,'date0',date0);
end

%Convert ice_eq depths to true depths for Gas Age Control points from best Realization
xx=[fcp2{end}(1,1):.1:fcp2{end}(end,1)*.99785+15.6622+5]; %fcp(z)*.99+15 from linear regression of depth v ice eq depth
FCP{MIS}=[interp1(ice_eq(xx,RICE_den(:,1),RICE_den(:,2)),xx,fcp2{end}(:,1)),... %Convert back to real depths
            fcp2{end}(:,2)]; %Ages of FCP in best simulation

%RMS error about Best realization
FCP_Err{MIS}=sqrt(sum((fcp2_interp_age(:,id)-fcp2{end}(:,2)*ones(1,sum(id))).^2,2)./sum(id));
%Area weighted mean RMS for all sample depths
FCP_WtErr{MIS}=trapz(fcp2{end}(:,2),FCP_Err{MIS}/range(fcp2{end}(:,2)));
%RMS for realizations with older/younger age estimates 
iii=fcp2_interp_age(:,id)>fcp2{end}(:,2)*ones(1,sum(id));
FCP_ul{MIS}=sqrt(sum((fcp2_interp_age(:,id).*iii-fcp2{end}(:,2)*ones(1,sum(id))).*iii).^2,2)./sum(iii,2));
FCP_ll{MIS}=sqrt(sum((fcp2_interp_age(:,id).*~iii-(fcp2{end}(:,2)*ones(1,sum(id))).*~iii).^2,2)./sum(~iii,2));
