library(ggplot2) 
#' @md
#' @title Simulate sediment archived proxy records from an "assumed true"
#'   climate signal.
#' @description \code{ClimToProxyClim} simulates the creation of a proxy climate
#'   record from a climate signal that is assumed to be true.
#'
#'   The following aspects of proxy creation are currently modelled.
#'
#'   1. Seasonal bias in the encoding of a proxy due to the interaction between
#'   climate seasonality and any seasonality in the life cycle of the organism
#'   encoding the climate signal (e.g. Foraminifera for Mg/Ca ratios, or
#'   phytoplankton for Alkenone unsaturation indices).
#'
#'   2. Bioturbation of the sediment archived proxy. For each requested
#'   timepoint, the simulated proxy consists of a weighted mean of the climate
#'   signal over a time window that is determined by the sediment accumulation
#'   rate \{sed.acc.rate} and the bioturbation depth \{bio.depth} which defaults
#'   to 10 cm. The weights are given by the depth solution to an impulse
#'   response function (Berger and Heath, 1968).
#'
#'   3. Aliasing of seasonal and inter-annual climate variation onto to
#'   bioturbated (smoothed) signal. For proxies measured on a small number of
#'   discrete particles both seasonal and inter-annual climate variation is
#'   aliased into the proxy record. For example, Foraminifera have a life-cycle
#'   of approximately 1 month, so they record something like the mean
#'   temperature from a single month. If Mg/Ca is measured on e.g.
#'   \code{n.samples} = 30 individuals, the measured proxy signal is a mean of
#'   30 distinct monthly mean temperatures and will thus be a stochastic sample
#'   of the true mean climate.
#'
#'   4. Measurement noise/error is added as a pure Gaussian white noise process
#'   with mean = 0, standard deviation = \code{meas.noise}.
#'
#'   5. Additionally, a random *bias* can be added to each realisation of a
#'   proxy record. Bias is simulated as a Gaussian random variable with mean =
#'   0, standard deviation = \code{meas.bias}. The same randomly generated bias
#'   value is applied to all timepoints in a simulated proxy record, when
#'   multiple replicate proxies are generated (\{n.replicates} > 1) each
#'   replicate has a different bias applied.
#'
#'   \code{ClimToProxyClim} returns one or more replicates of the final
#'   simulated proxy as well as several intermediate stages (see section
#'   **Value** below).
#'
#'
#' @param clim.signal The "assumed true" climate signal, e.g. climate model
#'   output or instrumental record. A \code{\link{ts}} object consisting of a
#'   years x 12 (months) x n habitats (e.g. depths) matrix of temperatures. The
#'   time series should be at annual resolution and in reverse, i.e. "most
#'   recent timepoint first" order.
#' @param timepoints The timepoints for which the proxy record is to be modelled

#' @param proxy.calibration.type Type of proxy, e.g. Uk'37 or MgCa, to which the
#'   clim.signal is converted before the archiving and measurement of the proxy
#'   is simulated
#' @param smoothed.signal.res The resolution, in years, of the smoothed (block
#'   averaged) version of the input climate signal returned for plotting. This
#'   does not affect what the proxy model uses as input. If set to NA, no
#'   smoothed climate output is generated, this can speed up some simulations.
#' @param proxy.prod.weights The seasonal pattern of productivity for the
#'   organism(s) archived in the proxy. A vector of values with length =
#'   ncols(clim.signal). i.e. 1 weight for each month x habitat combination.
#'   Defaults to a vector of equal weights.
#' @param bio.depth Depth of the bioturbated layer in cm, defaults to 10 cm. If
#'   bio.depth == 0, each timepoint samples from a single year of the
#'   clim.signal, equivalent to sampling a laminated sediment core.
#' @param sed.acc.rate Sediment accumulation rate in cm per 1000 years. Defaults
#'   to 50 cm per ka. Either a single value, or vector of same length as
#'   "timepoints"
#' @param layer.width the width of the sediment layer from which samples were
#'   taken, e.g. foraminifera were picked or alkenones were extracted, in cm.
#'   Defaults to 1 cm.
#' @param meas.noise The amount of noise to add to each simulated proxy value.
#'   Defined as the standard deviation of a normal distribution with mean = 0
#' @param meas.bias The amount of bias to add to each simulated proxy
#'   time-series. Each replicate proxy time-series has a constant bias added,
#'   drawn from a normal distribution with mean = 0, sd = meas.bias. Bias
#'   defaults to zero.
#' @param n.samples Number of e.g. Foraminifera sampled per timepoint, this can
#'   be either a single number, or a vector of length = timepoints
#' @param n.replicates Number of replicate proxy time-series to simulate from
#'   the climate signal
#'
#' @return \code{ClimToProxyClim} returns a list with three elements:
#'
#'   1. a dataframe \code{simulated.proxy}
#'   2. a dataframe \code{smoothed.signal}
#'   3. a dataframe \code{everything}
#'
#'
#'   The dataframe \code{simulated.proxy} contains a single realisation of the
#'   final forward modelled proxy, as well as the intermediate stages and the
#'   original climate signal at the requested timepoints.
#'
#'   The dataframe \code{smoothed.signal} contains a block averaged version the
#'   input climate signal, defaults to 100 year means but this is set by the
#'   parameter smoothed.signal.res. This is useful for plotting against the
#'   resulting simulated proxy.
#'
#'   The dataframe \code{everything} contains all of the above but with multiple
#'   replicates of the pseudo-proxy records if requested. The data are in
#'   "long form", with the column "stage" inidcating the proxy stage or input
#'   climate resolution and column "value" giving the values.
#'
#' **Named elements of the returned proxy record:**
#'
#' \describe{
#'    \item{timepoints}{Requested timepoints}
#'    \item{clim.signal.ann}{Input climate signal at requested timepoints at annual resolution}
#'    \item{clim.signal.smoothed}{Input climate signal at regular time intervals and resolution = smoothed.signal.res}
#'    \item{clim.timepoints.ssr}{Input climate signal at requested timepoints, smoothed to resolution = smoothed.signal.res}
#'    \item{proxy.bt}{Climate signal after bioturbation}
#'    \item{proxy.bt.sb}{Climate signal after bioturbation and production bias}
#'    \item{proxy.bt.sb.inf.b}{Climate signal after bioturbation, production bias, and calibration bias}
#'    \item{proxy.bt.sb.inf.b.n}{Climate signal after bioturbation, production bias, and measurement error}
#'    \item{proxy.bt.sb.sampY}{Climate signal after bioturbation, production bias, and aliasing of inter-annual variation}
#'    \item{proxy.bt.sb.sampYM}{Climate signal after bioturbation, production bias, and aliasing of inter-annual and intra-annual variation such as monthly temperatures or depth habitats}
#'    \item{proxy.bt.sb.sampYM.b}{Climate signal after bioturbation, production bias, and aliasing of inter-annual and intra-annual variation such as monthly temperatures or depth habitats, and calibration bias}
#'    \item{proxy.bt.sb.sampYM.b.n}{Climate signal after bioturbation, production bias, aliasing, and measurement error}
#'    \item{simulated.proxy}{Final simulated pseudo-proxy, this will be same as proxy.bt.sb.inf.b.n when n.samples = Inf, and proxy.bt.sb.sampYM.b.n when n.samples is finite}
#'    \item{observed.proxy}{True observed proxy (when supplied)}
#' }
#'
#'@importFrom dplyr tbl_df rename
#'@importFrom plyr alply
#'@export
#'
#' @examples
#' library(ggplot2)
#' set.seed(26052017)
#' clim.in <- ts(N41.t21k.climate[nrow(N41.t21k.climate):1,] - 273.15, start = -39)
#'
#' PFM <- ClimToProxyClim(clim.signal = clim.in,
#'                        timepoints = round(N41.proxy$Published.age),
#'                        proxy.calibration.type = "identity",
#'                        proxy.prod.weights = N41.G.ruber.seasonality,
#'                        sed.acc.rate = N41.proxy$Sed.acc.rate.cm.ka,
#'                        layer.width = 1,
#'                        meas.noise = 0.46, n.samples = Inf,
#'                        smoothed.signal.res = 10, meas.bias = 1,
#'                        n.replicates = 10)
#'
#' PlotPFMs(PFM$everything, max.replicates = 1, stage.order = "seq") +
#'   facet_wrap(~stage)
#'
#' PlotPFMs(PFM$everything, max.replicates = 1, stage.order = "var")
#'
#' PlotPFMs(PFM$everything, stage.order = "var", plot.stages = "all")
#'
ClimToProxyClim <- function(clim.signal,
                               timepoints,
                               proxy.calibration.type = c("identity", "UK37", "MgCa"),
                               smoothed.signal.res = 100,
                               proxy.prod.weights = rep(1/ncol(clim.signal),
                                               ncol(clim.signal)),
                               bio.depth = 10,
                               sed.acc.rate = 50,
                               layer.width = 1,
                               meas.noise = 0,
                               meas.bias = 0,
                               n.samples = Inf,
                               n.replicates = 1) {
  # Check inputs --------
  n.timepoints <- length(timepoints)

  if((length(n.samples) == 1 | length(n.samples)==n.timepoints)==FALSE)
    stop("n.sample must be either a single value, or a vector the same
         length as timepoints")

  if (all(is.finite(n.samples))==FALSE & all(is.infinite(n.samples))==FALSE)
    stop("n.samples cannot be a mix of finite and infinite")

  stopifnot(is.matrix(clim.signal))
  if (is.ts(clim.signal)==FALSE)
    stop("Since version 0.3.1 of sedproxy, ClimToProxyClim requires clim.signal to be a ts object")
  stopifnot(length(sed.acc.rate) == n.timepoints |
              length(sed.acc.rate) == 1)

  if (is.matrix(proxy.prod.weights))
    stop("Matrix form of seasonality not yet supported")

  # check no production weights match dimensions of climate
  #print(paste0("proxy.prod.weights = ", proxy.prod.weights))
  stopifnot(ncol(clim.signal) == length(proxy.prod.weights))

  # Ensure seasonal productivities are weights
  proxy.prod.weights <- proxy.prod.weights / sum(proxy.prod.weights)

  # Calculate timepoint invariant values ------
  max.clim.signal.i <- end(clim.signal)[1]
  min.clim.signal.i <- start(clim.signal)[1]

  if (length(sed.acc.rate) == 1) {
    sed.acc.rate <- rep(sed.acc.rate, n.timepoints)
  }

  # Replicate n.samples if not vector
  if (length(n.samples) == 1) {
    n.samples <- rep(n.samples, n.timepoints)
  }

  # Check whether bioturbation window will extend beyond climate signal for any of the timepoints

  # bioturbation window will be focal.timepoint - bio.depth.timesteps - layer.width.years / 2 to
  # focal.timepoint + 3*bio.depth.timesteps

  max.min.windows <- matrix(t(sapply(1:length(timepoints), function(tp) {
    bio.depth.timesteps <- round(1000 * bio.depth / sed.acc.rate[tp])
    layer.width.years <- ceiling(1000 * layer.width / sed.acc.rate[tp])
    return(c(max = timepoints[tp] + 3 * bio.depth.timesteps,
             min = timepoints[tp] - bio.depth.timesteps - layer.width.years / 2))
  })), ncol = 2)

  colnames(max.min.windows) <- c("max", "min")

  #print(max.min.windows)
  max.ind <- max.min.windows[,"max"] >= max.clim.signal.i
  min.ind <- max.min.windows[,"min"] <  min.clim.signal.i

  if (any(max.ind))
    warning(paste0("One or more requested timepoints is too old. Bioturbation window(s) for timepoint(s) ",
                   paste(timepoints[max.ind], collapse = ", "),
                   " extend(s) beyond end of input climate signal. Returning pseudo-proxy for valid timepoints."))

  if (any(max.min.windows[,"min"] < min.clim.signal.i))
    warning(paste0("One or more requested timepoints is too recent. Bioturbation window(s) for timepoint(s) ",
                   timepoints[max.min.windows[, "min"] < min.clim.signal.i],
                   " extend(s) above start of input climate signal. Returning pseudo-proxy for valid timepoints."))


  timepoints <- timepoints[max.ind == FALSE & min.ind == FALSE]
  n.timepoints <- length(timepoints)

  # Remove timepoints that exceed clim.signal ------
  sed.acc.rate <- sed.acc.rate[max.ind == FALSE & min.ind == FALSE]
  n.samples <- n.samples[max.ind == FALSE & min.ind == FALSE]
  max.min.windows <- max.min.windows[max.ind == FALSE & min.ind == FALSE, , drop = FALSE]


  # Convert to proxy units if requested --------
  proxy.calibration.type <- match.arg(proxy.calibration.type)

  if (proxy.calibration.type != "identity") {
    mean.temperature <-  mean(as.vector(clim.signal))
    proxy.clim.signal <-
      matrix(
        ProxyConversion(
          temperature = as.vector(clim.signal),
          proxy.calibration.type = proxy.calibration.type,
          point.or.sample = "point",
          n = 1
        )[, 1],
        ncol = ncol(clim.signal),
        byrow = FALSE
      )
    meas.noise <- as.vector(ProxyConversion(temperature = mean.temperature + meas.noise,
                                            proxy.calibration.type = proxy.calibration.type) -
                              ProxyConversion(temperature = mean.temperature,
                                              proxy.calibration.type = proxy.calibration.type))
  } else{
    proxy.clim.signal <- clim.signal
  }

  # Create smoothed climate signal --------
  if (is.na(smoothed.signal.res)) {
    timepoints.smoothed <- NA
    clim.signal.smoothed <- NA
  } else{
    timepoints.smoothed <- seq(min.clim.signal.i, max.clim.signal.i, by = smoothed.signal.res)
    clim.signal.smoothed <- ChunkMatrix(timepoints.smoothed, smoothed.signal.res,
                                        proxy.clim.signal)
  }


  # For each timepoint ------
  out <- sapply(1:n.timepoints, function(tp) {
    # Get bioturbation window ----------
    first.tp <- max.min.windows[tp, "min"]
    last.tp <- max.min.windows[tp, "max"]
    bioturb.window <- first.tp:last.tp

    # Get bioturbation weights --------
    bioturb.weights <- BioturbationWeights(z = bioturb.window, focal.depth = timepoints[tp],
                                           layer.width = layer.width, sed.acc.rate = sed.acc.rate[tp],
                                           bio.depth = bio.depth)

    bioturb.weights <- bioturb.weights / sum(bioturb.weights)

    # Get portion of clim.signal corresponding to bioturbation window -------
    ## window is slow
    # clim.sig.window <- stats:::window.ts(proxy.clim.signal,
    #                           start = first.tp,
    #                           end = last.tp)
    clim.sig.window <-  clim.signal[first.tp:last.tp - min.clim.signal.i+1, ]

    # this is estimating mean deviation MD, (not MAD or SD)
    # no need to estimate this from the psuedo data
    # MD = 2/(exp(1)/std) for exponential, where std = lambda = bio.depth.timesteps
    # smoothing.width = sum(bioturb.weights*abs(bioturb.window))

    # Get bioturbation X no-seasonality weights matrix ---------
    biot.sig.weights <- bioturb.weights %o% rep(1, ncol(clim.signal))
    biot.sig.weights <- biot.sig.weights / sum(biot.sig.weights)

    # Get bioturbation X seasonality weights matrix ---------
    clim.sig.weights <- bioturb.weights %o% proxy.prod.weights
    clim.sig.weights <- clim.sig.weights / sum(clim.sig.weights)

    # Check weights sum to 1, within tolerance
    weight.err <- abs(sum(clim.sig.weights) - 1)
    if ((weight.err < 1e-10) == FALSE) stop(paste0("weight.err = ", weight.err))


    # Calculate mean clim.signal -------

    # Just bioturbation
    proxy.bt <- sum(biot.sig.weights * clim.sig.window)

    # Bioturbation + seasonal bias
    proxy.bt.sb <- sum(clim.sig.weights * clim.sig.window)

    # Bioturbation + seasonal bias + aliasing
    if (is.infinite(n.samples[tp])) {
      proxy.bt.sb.sampY <- rep(NA, n.replicates)
      proxy.bt.sb.sampYM <- rep(NA, n.replicates)
    } else if (is.finite(n.samples[tp])) {
      # call sample once for all replicates together, then take means of
      # groups of n.samples
      # Get indices not values
      samp.indices <-  sample(length(clim.sig.window),
                              n.samples[tp] * n.replicates,
                              prob = clim.sig.weights,
                              replace = TRUE)

      # convert vector to matrix (cheap only attributes changed), then means
      # can be taken across columns to get per replicate means
      samp <- matrix(clim.sig.window[samp.indices], nrow = n.samples[tp])
      #proxy.bt.sb.sampYM <- apply(samp, 2, mean)
      proxy.bt.sb.sampYM <- colMeans(samp)

      # Get without seasonal aliasing (bioturbation aliasing only)
      clim.sig.window.ann <- rowSums(clim.sig.window %*% diag(proxy.prod.weights))

      # weights passed as a matrix are applied columnwise, so
      # modulo on nrows is need here
      row.indices <- (samp.indices-1) %% nrow(clim.sig.window) + 1

      samp.bt <- matrix(clim.sig.window.ann[row.indices], nrow = n.samples[tp])
      proxy.bt.sb.sampY <- colMeans(samp.bt)

    }


    # Gather output ----------
    list(
      #smoothing.width = smoothing.width,
      proxy.bt = proxy.bt,
      proxy.bt.sb = proxy.bt.sb,
      proxy.bt.sb.sampY = proxy.bt.sb.sampY,
      proxy.bt.sb.sampYM = proxy.bt.sb.sampYM)
  })

  #out <- apply(out, 1, function(x) simplify2array(x))
  out <- plyr::alply(out, 1, function(x) simplify2array(x), .dims = TRUE)

  # remove extra attributes added by alply
  attr(out, "split_type") <- NULL
  attr(out, "split_labels") <- NULL

  #print(out$proxy.bt.sb.sampYM)
  if (n.replicates == 1) out$proxy.bt.sb.sampYM <- matrix(out$proxy.bt.sb.sampYM, nrow = 1)
  out$proxy.bt.sb.sampYM <- t(out$proxy.bt.sb.sampYM)

  if (n.replicates == 1) out$proxy.bt.sb.sampY <- matrix(out$proxy.bt.sb.sampY, nrow = 1)
  out$proxy.bt.sb.sampY <- t(out$proxy.bt.sb.sampY)
  #print(out$proxy.bt.sb.sampYM)

  # Add bias and noise to infinite sample --------
  if (meas.bias != 0) {
    bias <- stats::rnorm(n = n.replicates, mean = 0, sd = meas.bias)
  } else{
    bias <- rep(0, n.replicates)
  }
  if (meas.noise != 0) {
    noise <- stats::rnorm(n = n.replicates * n.timepoints, mean = 0, sd = meas.noise)
  }else{
    noise <- rep(0, n.replicates)
  }

  out$proxy.bt.sb.inf.b <- outer(out$proxy.bt.sb, bias, FUN = "+")
  out$proxy.bt.sb.inf.b.n <- out$proxy.bt.sb.inf.b + noise

  if (all(is.finite(n.samples))){
    out$proxy.bt.sb.inf.b[,] <- NA
    out$proxy.bt.sb.inf.b.n[,] <- NA
  }

  # Add bias and noise to finite sample --------
  out$proxy.bt.sb.sampYM.b <- out$proxy.bt.sb.sampYM + bias
  out$proxy.bt.sb.sampYM.b.n <- out$proxy.bt.sb.sampYM.b + noise

  # set intermediate bias stages to NA if no bias modelled
  if (meas.bias == 0) {
    out$proxy.bt.sb.inf.b[,] <- NA
    out$proxy.bt.sb.sampYM.b[,] <- NA
  }

  # Calculate chunked climate at timepoints

  # Create smoothed climate signal
  if (is.na(smoothed.signal.res)) {
    out$clim.timepoints.ssr <- NA

  } else{
    out$clim.timepoints.ssr <- ChunkMatrix(timepoints, smoothed.signal.res, proxy.clim.signal)
  }

  # Add items to output list -----------
  out$timepoints = timepoints
  out$clim.signal.ann = rowSums(proxy.clim.signal[timepoints,  , drop = FALSE]) / ncol(proxy.clim.signal)
  #out$sed.acc.rate = sed.acc.rate
  out$timepoints.smoothed = timepoints.smoothed
  out$clim.signal.smoothed = clim.signal.smoothed

  # Organise output -------
  simulated.proxy <-
    dplyr::tbl_df(out[c(
      "timepoints",
      "clim.signal.ann",
      "clim.timepoints.ssr",
      "proxy.bt",
      "proxy.bt.sb"#,
      #"sed.acc.rate",
      #"smoothing.width"
    )])

  simulated.proxy$proxy.bt.sb.sampY <- out$proxy.bt.sb.sampY[, 1, drop = TRUE]
  simulated.proxy$proxy.bt.sb.sampYM <- out$proxy.bt.sb.sampYM[, 1, drop = TRUE]
  simulated.proxy$proxy.bt.sb.inf.b <- out$proxy.bt.sb.inf.b[, 1, drop = TRUE]
  simulated.proxy$proxy.bt.sb.inf.b.n <- out$proxy.bt.sb.inf.b.n[, 1, drop = TRUE]
  simulated.proxy$proxy.bt.sb.sampYM.b <- out$proxy.bt.sb.sampYM.b[, 1, drop = TRUE]
  simulated.proxy$proxy.bt.sb.sampYM.b.n <- out$proxy.bt.sb.sampYM.b.n[, 1, drop = TRUE]

  if (all(is.finite(n.samples))) {
    simulated.proxy$simulated.proxy <- simulated.proxy$proxy.bt.sb.sampYM.b.n
    out$simulated.proxy <- out$proxy.bt.sb.sampYM.b.n
  } else{
    simulated.proxy$simulated.proxy <- simulated.proxy$proxy.bt.sb.inf.b.n
    out$simulated.proxy <- out$proxy.bt.sb.inf.b.n
  }


  smoothed.signal <- dplyr::tbl_df(out[c(
    "timepoints.smoothed",
    "clim.signal.smoothed"
  )])

  smoothed.signal <- dplyr::rename(smoothed.signal,
                                   timepoints = timepoints.smoothed,
                                   value = clim.signal.smoothed)

  smoothed.signal$stage <- "clim.signal.smoothed"

  everything <- MakePFMDataframe(out)

  return(list(simulated.proxy=simulated.proxy,
              smoothed.signal=smoothed.signal,
              everything = everything))
  #return(everything)
}

ChunkMatrix <- function(timepoints, width, climate.matrix){

  if (is.ts(climate.matrix)) {
     sapply(timepoints, function(tp){
      rel.wind <- 1:width -round(width/2)
      #if(tp == 10000) print(rel.wind)
      inds <- rel.wind + tp - start(climate.matrix)[1] + 1
      #if(tp == 10000) print(inds)
      inds <- inds[inds > 0 & inds < nrow(climate.matrix)]
      #if(tp == 10000) print(inds)
      m <- climate.matrix[inds, , drop = FALSE]
      mean(m)
    })}else{
    max.clim.signal.i <- nrow(climate.matrix)

    rel.wind <- 1:width -round(width/2)

    sapply(timepoints, function(tp){

      avg.window.i.1 <- (rel.wind) + tp

      if (max(avg.window.i.1) > max.clim.signal.i) {
        warning("In ChunkMatrix: window extends below end of clim.signal")
      }

      avg.window.i <- avg.window.i.1[avg.window.i.1 > 0 &
                                       avg.window.i.1 < max.clim.signal.i]

      stopifnot(avg.window.i > 0)
      stopifnot(max.clim.signal.i > max(avg.window.i))
      #if(tp == 10000) print(avg.window.i)
      mean(climate.matrix[avg.window.i, , drop = FALSE])
    })}
}


#' Convert "everything" part of output from ClimToProxyClim to dataframe.
#' Used internally.
#'
#' @param PFM output from ClimToProxyClim
#' @return a dataframe
#' @importFrom dplyr bind_rows filter
#' @importFrom tidyr gather
MakePFMDataframe <- function(PFM){
  df <- data.frame(
    proxy.bt.sb.sampY = as.vector(PFM$proxy.bt.sb.sampY),
    proxy.bt.sb.sampYM = as.vector(PFM$proxy.bt.sb.sampYM),
    proxy.bt.sb.inf.b = as.vector(PFM$proxy.bt.sb.inf.b),
    proxy.bt.sb.sampYM.b = as.vector(PFM$proxy.bt.sb.sampYM.b),
    proxy.bt.sb.inf.b.n = as.vector(PFM$proxy.bt.sb.inf.b.n),
    proxy.bt.sb.sampYM.b.n = as.vector(PFM$proxy.bt.sb.sampYM.b.n),
    simulated.proxy = as.vector(PFM$simulated.proxy),
    stringsAsFactors = FALSE)

  df$timepoints <- PFM$timepoints
  df$replicate <- rep(1:ncol(PFM$proxy.bt.sb.inf.b), each = length(PFM$timepoints))
  df <- dplyr::tbl_df(df)
  df <- tidyr::gather(df, stage, value, -timepoints, -replicate)

  df2 <- data.frame(
    replicate = 1,
    timepoints = PFM$timepoints,
    proxy.bt = PFM$proxy.bt,
    proxy.bt.sb = PFM$proxy.bt.sb,
    clim.signal.ann = PFM$clim.signal.ann,
    clim.timepoints.ssr = PFM$clim.timepoints.ssr,
    stringsAsFactors = FALSE)
  df2 <- tidyr::gather(df2, stage, value, -timepoints, -replicate)

  df.smoothed <- data.frame(
    replicate = 1,
    timepoints = PFM$timepoints.smoothed,
    stage = "clim.signal.smoothed",
    value = PFM$clim.signal.smoothed,
    stringsAsFactors = FALSE)

  rtn <- dplyr::bind_rows(df, df2, df.smoothed)

  rtn <- droplevels(dplyr::filter(rtn, stats::complete.cases(value)))
  rtn <- dplyr::left_join(rtn, stages.key, by = "stage")

  return(rtn)
}
#' Plot forward modelled sedimentary proxies
#'
#' @param PFMs A dataframe of forward modelled proxies
#' @param stage.order Controls the order in which proxy stages are plotted,
#' either sequentially, "seq", or in order of variance, "var". Defaults to var.
#' @param plot.stages Proxy stages to be plotted, "default", "all", or a custom character vector
#' @param colr.palette Colours for the proxy stages
#' @param alpha.palette Alpha levels for the proxy stages
#' @param levl.labels Labels for the proxy stages
#' @param max.replicates Maximum number of replicates to plot at once
#'
#' @import ggplot2
#' @importFrom dplyr filter
#' @export PlotPFMs
#'
#' @examples
#' library(ggplot2)
#' set.seed(26052017)
#' clim.in <- N41.t21k.climate[nrow(N41.t21k.climate):1,] - 273.15
#'
#' PFM <- ClimToProxyClim(clim.signal = clim.in,
#'                        timepoints = round(N41.proxy$Published.age),
#'                        proxy.calibration.type = "identity",
#'                        proxy.prod.weights = N41.G.ruber.seasonality,
#'                        sed.acc.rate = N41.proxy$Sed.acc.rate.cm.ka,
#'                        meas.noise = 0.46, n.samples = Inf,
#'                        smoothed.signal.res = 10, meas.bias = 1,
#'                        n.replicates = 10)
#'
#' PlotPFMs(PFM$everything, max.replicates = 1, stage.order = "seq") +
#'   facet_wrap(~stage)
#'
#' PlotPFMs(PFM$everything, max.replicates = 1, stage.order = "var")
#'
#' PlotPFMs(PFM$everything, stage.order = "var", plot.stages = "all")
#'
PlotPFMs <- function(PFMs,
                     stage.order = c("var", "seq"),
                     plot.stages = c("default"),
                     max.replicates = 5,
                     colr.palette = "default",
                     alpha.palette = "default",
                     levl.labels = "default"){

  if(exists("replicate", where = PFMs)){
    rug.dat <- dplyr::filter(PFMs, stage %in% c("simulated.proxy", "observed.proxy"),
                             replicate == 1)
  }else{
    rug.dat <- dplyr::filter(PFMs, stage %in% c("simulated.proxy", "observed.proxy"))
    rug.dat$replicate <- 1
    PFMs$replicate <- 1
  }

  if(exists("Location", where = PFMs)==FALSE){
    PFMs$Location <- ""
  }

  if(exists("ID.no", where = PFMs)==FALSE){
    PFMs$ID.no <- ""
  }
  if(exists("Proxy", where = PFMs)==FALSE){
    PFMs$Proxy <- ""
  }

  # assign default asthetic mappings

  breaks <- stages.key$stage

  if (colr.palette[1] == "default")
    colr.palette  <-
      structure(stages.key$plotting.colour,
                .Names = stages.key$stage)

  if (alpha.palette[1] == "default") alpha.palette  <-
      structure(stages.key$plotting.alpha,
                .Names = stages.key$stage)

  if (levl.labels[1] == "default") levl.labels  <-
      structure(stages.key$label,
                .Names = stages.key$stage)

  if (plot.stages[1] == "default") {
    plotting.levels <- c(
      "clim.signal.smoothed", "proxy.bt", "proxy.bt.sb",
      "proxy.bt.sb.sampYM",  "simulated.proxy", "observed.proxy"
      )
  } else if (plot.stages == "all") {
    plotting.levels <- stages.key$stage
    plotting.levels <- subset(plotting.levels, plotting.levels %in% c("clim.signal.ann", "clim.timepoints.ssr") == FALSE)
  } else{
    plotting.levels <- plot.stages
  }

  PFMs <- dplyr::filter(PFMs, stage %in% plotting.levels,
                        replicate <= max.replicates)
  #set factor level ordering for stages
  stage.order <- match.arg(stage.order)
  switch(stage.order,
         seq = PFMs$stage <- factor(PFMs$stage, levels = plotting.levels, ordered = TRUE),
         var = {
           var.order <- tapply(PFMs$value, PFMs$stage, FUN = var)
           var.order <- rank(var.order, ties.method = "first")
           var.order <- names(sort(var.order, decreasing = TRUE))
           PFMs$stage <- factor(PFMs$stage,
                                levels = var.order, ordered = TRUE)
           })


  p <- ggplot2::ggplot(data = PFMs, aes(x = timepoints, y = value,
                               colour = stage, alpha = stage,
                               linetype = as.factor(replicate))) +
    #geom_rug(data = rug.dat, sides = "b", colour = "Darkgrey") +
    geom_line() +
    theme_bw() +
    theme(legend.position = "top", panel.grid.minor = element_blank()) +
    guides(colour = guide_legend(label.position = "top",
                                 label.hjust = 1,
                                 nrow = 1,
                                 override.aes = list(alpha = 1))) +
    labs(x = expression("Timepoints"),
         y = expression("Proxy value")) +
    scale_linetype_manual(values = rep(1, 13), guide = FALSE)+
    scale_alpha_manual(guide = FALSE)

  if (is.null(colr.palette) == FALSE)
    p <- p + scale_colour_manual("", values = colr.palette, breaks = names(colr.palette),
                                 labels = levl.labels)

  if (is.null(alpha.palette) == FALSE)
    p <- p + scale_alpha_manual("", values = alpha.palette, breaks = names(alpha.palette),
                                labels = levl.labels)

  return(p)
}






#' Bioturbation weights
#' @description For a given focal depth (or time), this function returns the probability
#' that material collected from that depth was orignially deposited at depth(s)
#' z. In other words, that the material would have been found at depth z if there
#' had been no bioturbation. It is the convolution of the depth solution from
#' Berger and Heath (1968) with a uniform distribution to account for the width
#' of the sediment layer from which samples
#' were picked/extracted. It is a probability density function.
#' @inheritParams ClimToProxyClim
#' @param z A vector of times or depths at which to evaluate the bioturbation weights
#' @param focal.depth The depth (or time) for which source dates are wanted
#' @param scale whether to scale depths by sediment accumulation rate to give
#' positions in terms of time
#' @return a vector of weights
#' @export
#' @references Berger, W. H., & Heath, G. R. (1968).
#' Vertical mixing in pelagic sediments.
#' Journal of Marine Research, 26(2), 134–143.
#' @examples
#' z <- 0:10000
#' w <- BioturbationWeights(z, focal.depth = 2000, layer.width = 1, sed.acc.rate = 5 / 1000, bio.depth = 10)
#' plot(z, w, "l")
BioturbationWeights <- function(z, focal.depth, layer.width=1, sed.acc.rate, bio.depth, scale = c("time", "depth")){

  sed.acc.rate <- sed.acc.rate / 1000

  scale <- match.arg(scale)

  if (scale == "time"){
    lwy <- (layer.width / sed.acc.rate)
    mdy <- (bio.depth / sed.acc.rate)
  }else{
    lwy <- (layer.width)
    mdy <- (bio.depth)
  }

  fd <- focal.depth

  C <- lwy/2
  lam <- 1/mdy

  z <- z - fd + mdy

  if (mdy <= 1){
    fz <- dunif(z, -C, C)
  }else if (lwy == 0){
    fz <- dexp(z, 1/mdy)
  }else{
    fz <- (z < -C) * 0 +
      (z >= -C & z <= C) * (lam*(1/lam-exp(-lam*C-lam*z)/lam))/(2*C)  +
      (z > C) * (lam*(exp(lam*C-lam*z)/lam-exp(-lam*C-lam*z)/lam))/(2*C)
  }

  return(fz)
}
# Objects
stages.key <- structure(list(stage = c("timepoints", "clim.signal.ann", "clim.signal.smoothed",
                         "clim.timepoints.ssr", "proxy.bt", "proxy.bt.sb", "proxy.bt.sb.inf.b",
                         "proxy.bt.sb.inf.b.n", "proxy.bt.sb.sampY", "proxy.bt.sb.sampYM",
                         "proxy.bt.sb.sampYM.b", "proxy.bt.sb.sampYM.b.n", "simulated.proxy",
                         "observed.proxy"), label = c("Requested timepoints", "(1) Input climate",
                                                      "(1) Input climate", "(1) Input climate", "(2) +Bioturbation",
                                                      "(3) +Production bias", "(.) +Calibration bias", "(5) +Measurement error",
                                                      "(4) +Aliasing Y", "(4) +Aliasing YM", "(.) +Calibration bias",
                                                      "(5) +Measurement error", "(5) Simulated proxy", "(*) Observed proxy"
                         ), description = c("Requested timepoints", "Input climate signal at requested timepoints at annual resolution",
                                            "Input climate signal at regular time intervals and resolution = smoothed.signal.res",
                                            "Input climate signal at requested timepoints, smoothed to resolution = smoothed.signal.res",
                                            "Climate signal after bioturbation", "Climate signal after bioturbation and production bias",
                                            "Climate signal after bioturbation, production bias, and calibration bias",
                                            "Climate signal after bioturbation, production bias, and measurement error",
                                            "Climate signal after bioturbation, production bias, and aliasing of inter-annual variation",
                                            "Climate signal after bioturbation, production bias, and aliasing of inter-annual and intra-annual variation such as monthly temperatures or depth habitats",
                                            "Climate signal after bioturbation, production bias, and aliasing of inter-annual and intra-annual variation such as monthly temperatures or depth habitats, and calibration bias",
                                            "Climate signal after bioturbation, production bias, aliasing, and measurement error",
                                            "Final simulated pseudo-proxy, this will be same as proxy.bt.sb.inf.b.n when n.samples = Inf, and proxy.bt.sb.sampYM.b.n when n.samples is finite",
                                            "True observed proxy (when supplied)"), plot.order = c(1, 1,
                                                                                                   2, 3, 4, 5, 10, 6, 7, 8, 9, 11, 12, 13), plotting.colour = c("Black",
                                                                                                                                                                "#018571", "#018571", "#018571", "Green", "Gold", "Pink", "#7570b3",
                                                                                                                                                                "#d95f02", "#d95f02", "Pink", "#7570b3", "#7570b3", "Red"), plotting.alpha = c(1,
                                                                                                                                                                                                                                               1, 1, 1, 1, 1, 1, 1, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5)), .Names = c("stage",
                                                                                                                                                                                                                                                                                                               "label", "description", "plot.order", "plotting.colour", "plotting.alpha"
                                                                                                                                                                                                                                               ), class = c("tbl_df", "tbl", "data.frame"), row.names = c(NA,
                                                                                                                                                                                                                                                                                                          -14L))

# Functions
SimPowerlaw <- function(beta, N)
{
  N2 <- (3 ^ ceiling(log(N, base = 3)))
  df  <- 1 / N2
  f <- seq(from = df, to = 1 / 2, by = df)
  Filter <- sqrt(1 / (f ^ beta))
  Filter <- c(max(Filter), Filter, rev(Filter))
  x   <- rnorm(N2, 1)
  fx  <- fft(x)
  ffx <- fx * Filter
  result <- Re(fft(ffx, inverse = TRUE))[1:N]
  return(scale(result)[1:N])
}

# Define UI ----
ui <- fluidPage(
  titlePanel("sedproxy"),
  p(
    em("sedproxy"),
    "is a forward model for sediment archived climate proxies.
    It is based on work described in Laepple and Huybers (2013).
    A manuscript is in preparation, Dolman and Laepple (in prep.),
    which will more fully describe the forward model and its applications.
    Please contact Dr Andrew Dolman <andrew.dolman@awi.de>,
    or Dr Thomas Laepple <tlaepple@awi.de>, at the Alfred-Wegener-Institute,
    Helmholtz Centre for Polar and Marine Research,
    Germany, for more information.
    "
  ),
  p("This work was supported by German Federal Ministry of Education and Research
    (BMBF) as Research for Sustainability initiative", a("(FONA)", href = "https://www.fona.de/",  target = "_blank"),
    "through the", a("PalMod", href = "https://www.palmod.de/",  target = "_blank"), "project (FKZ: 01LP1509C).",
    br(), br(),
    a(img(src="PalMod_Logo_RGB.png", align = "top"),
      href = "https://www.palmod.de/",
      target = "_blank")), br(),
  p(
    "Reference: ",
    "Laepple, T., & Huybers, P. (2013): Reconciling discrepancies between Uk37
    and Mg/Ca reconstructions of Holocene marine temperature variability.
    Earth and Planetary Science Letters, 375: 418-429."
  ),
  sidebarPanel(tabsetPanel(
    tabPanel(
      "Model parameters",
      fluidRow(
        h4(
          "Update the parameter values below
          and then run the proxy forward model."
        ),
        column(12,
               actionButton("run.pfm", "Run forward model")),
        hr()
        ),
      fluidRow(
        h4("Setup input climate signal"),
        column(
          width = 12,
          sliderInput(
            "clim.signal.length",
            h5("Length of input climate signal [years]"),
            value = 25000,
            step = 1000,
            min = 5000,
            max = 100000
          )
        ),
        column(
          width = 6,
          numericInput(
            "clim.signal.beta",
            h5("Slope of the power spectrum of the input climate signal"),
            value = 1,
            step = 0.1,
            min = 0.1,
            max = 3
          )
        ),
        column(
          width = 6,
          numericInput(
            "seas.amp",
            h5("Amplitude of the seasonal cycle"),
            value = 5,
            step = 0.5,
            min = 0,
            max = 20
          )
        )
      ),
      fluidRow(
        h4("Control sampling"),
        column(width = 6,
               numericInput(
                 "seed",
                 h5("Set RNG seed"),
                 value = 1,
                 step = 1,
                 min = 1
               )),
        column(
          width = 6,
          numericInput(
            "n.replicates",
            h5("No. replicates"),
            value = 1,
            step = 1,
            min = 1,
            max = 100
          )
        )
      ),
      fluidRow(
        column(
          width = 6,
          numericInput(
            "t.res",
            h5("Core sampling resolution [years]"),
            value = 100,
            step = 100,
            min = 1,
            max = 10000
          )
        ),
        column(
          width = 6,
          numericInput(
            "n.samples",
            h5("No. samples per timepoint"),
            value = 30,
            step = 1,
            min = 1,
            max = 1000
          )
        )
      ),
      fluidRow(
        h4("Sedimentation parameters"),
        column(
          6,
          numericInput(
            "bio.depth",
            h5("Bioturbation depth [cm]"),
            value = 10,
            step = 1,
            min = 0,
            max = 30
          )
        ),
        column(
          6,
          numericInput(
            "sed.acc.rate",
            h5("Sediment accumulation rate [cm/ka]"),
            value = 50,
            step = 1,
            min = 0,
            max = 100
          )
        )
      ),
      fluidRow(h4("Proxy production weights (monthly)"),
               column(12,
                      fluidRow(
                        column(
                          12,
                          radioButtons(
                            "seas",
                            label = NULL,
                            choices = c("Uniform", "Custom"),
                            selected = "Uniform",
                            inline = TRUE
                          ),
                          conditionalPanel(
                            condition = "input.seas == 'Custom'",
                            textInput(
                              "mon.vec",
                              "Modify the 12 monthly weights",
                              "1,1,1,1,1,1,1,1,1,1,1,1"
                            ),
                            span(textOutput("proxy.prod.weights.check"), style = "color:red")
                          )
                        )
                      ))),
      fluidRow(
        h4("Noise parameters"),
        column(
          6,
          numericInput(
            "meas.noise",
            h5("Measurement noise"),
            value = 0.46,
            step = 0.01,
            min = 0,
            max = 1
          )
        ),
        column(
          6,
          numericInput(
            "meas.bias",
            h5("Measurement bias"),
            value = 0,
            step = 0.1,
            min = 0,
            max = 2
          )
        )
      )

  ),
  tabPanel("Plot appearance",
           fluidRow(
             h4("Plot proxy stages:"),
             checkboxGroupInput(
               "stages",
               "Stages:",
               choices = list(
                 "Input climate" = "clim.signal.smoothed",
                 "Bioturbated climate" = "proxy.bt",
                 "Bioturbated + seasonally biased climate" =  "proxy.bt.sb",
                 "Sampled climate inc. aliasing" = "proxy.bt.sb.sampYM",
                 "Final pseudo-proxy" = "simulated.proxy"
               ),
               selected = list(
                 "clim.signal.smoothed",
                 "proxy.bt",
                 "proxy.bt.sb",
                 "proxy.bt.sb.sampYM",
                 "simulated.proxy"
               )
             )
           ))
    )),
  mainPanel(tabsetPanel(
    tabPanel("Plots",
             plotOutput("pfm.plot", height = "800px")),
    tabPanel("Numbers",
             dataTableOutput("pfm.str")),
    tabPanel("Placeholder", textOutput("proxy.prod.weights"))
  ))
  )


# Define server logic ----
server <- function(input, output) {
  clim <- eventReactive(input$run.pfm, {
    set.seed(input$seed)
    ann <-
      SimPowerlaw(input$clim.signal.beta, input$clim.signal.length)
    mon <-
      cos(seq(pi, 3 * pi, length.out = 12)) * input$seas.amp / 2
    clim <- outer(ann, mon, "+")
    clim <- ts(clim, start = 1)
    return(clim)
  }, ignoreNULL = FALSE)
  timepoints <- eventReactive(input$run.pfm, {
    #res <- 100
    tp <- seq(1, input$clim.signal.length, by = input$t.res)
    t.min <-
      ceiling(1000 * input$bio.depth / input$sed.acc.rate) + 1
    t.max <- input$clim.signal.length - 3 * t.min
    tp <- tp[tp > t.min & tp < t.max]
    return(tp)
  }, ignoreNULL = FALSE)
  seasprod <- eventReactive({
    input$mon.vec
    input$seas
  }, {
    if (input$seas == 'Custom')
    {
      v <- as.numeric(unlist(strsplit(input$mon.vec, ",")))
    } else{
      v <- rep(1, 12)
    }
    return(v)
  }, ignoreNULL = FALSE)
  output$proxy.prod.weights.check <- renderText({
    if (length(seasprod()) != 12)
    {
      paste0("You entered ",
             length(seasprod()),
             " values; 12 are required.")
    }
  })
  pfm <- eventReactive(input$run.pfm, {
    pfm <- ClimToProxyClim(
      clim.signal = clim(),
      timepoints = timepoints(),
      smoothed.signal.res = 100,
      bio.depth = input$bio.depth,
      sed.acc.rate = input$sed.acc.rate,
      proxy.prod.weights = seasprod(),
      n.samples = input$n.samples,
      n.replicates = input$n.replicates,
      meas.noise = input$meas.noise,
      meas.bias = input$meas.bias
    )
  }, ignoreNULL = FALSE)
  output$pfm.plot <- renderPlot({
    dat <- pfm()$everything
    dat <- subset(dat, dat$stage %in% input$stages)
    if (nrow(dat) > 0) {
      PlotPFMs(dat) +
        ggplot2::labs(x = "Age [years]")
    }
  }, res = 72 * 2)
  output$pfm.str <- renderDataTable({
    round(pfm()$simulated.proxy, 5)
  })
}

# Run the app ----
shinyApp(ui = ui, server = server)

