MoCo <- function(file_climate = "MoCo_climate.txt",
                 file_param = "",
                 param = c(-4.34,19.73,5.7,1.04,0.969,80,10.48,1,20,1,22,6,rep(1,12),0.1,0.1,0.08,1.5,1000),
                 verbose = TRUE){
# MoCo program (Mollucs and Corals)
# Estimate the the systematic and standard errors of paleoclimate
# reconstructions based on mollusk or corals geochemical proxies
#
# Arguments:
#      file_climate : the name of the file in which monthly climatic data
#                     are stored. The monthly values must be in the last 
#                     column
#      file_param   : the name of the file where the 29 parameter values are 
#                     stored, in the order specified below. 
#                     Each line must begin with the parameter value. The 
#                     comment character is "#". Parameters can be passed 
#                     alternatively as a vector (see param)
#      param        : vector of length 29 containing the parameter values 
#                     in the order speciefied below. Parameters can be passed
#                     alternatively as a parameter file (see file_param)
#      verbose      : boolean, if TRUE, displays the results of MoCo
#
# Details:
#     Parameter should be in this order. 
#     The first seven parameters relate to the dataset of the 
#     proxy linear model calibraton (temperature T, proxy P, T= alpha.P + beta)
#     Default values correspond to the Grossmann and Ku's (Chemical Geology, 
#     1986, v.59, p59-74) paleotemperature for aragonite (All data), 
#     with delta18O(water) expressed versus SMOW.
#       param[1] = alpha
#       param[2] = beta
#       param[3] = sigmaT,  standard deviation of temperature values 
#                           in calibration dataset
#       param[4] = sigmaP,  standard deviation of proxy values 
#                           in calibration dataset
#       param[5] = r,       correlation coefficient of Ti and Pi
#       param[6] = ncalib,  number of calibration data points, must be >= 3
#       param[7] = T0,      mean value of T in calibration dataset
#     The following 17 parameters are related to biological properties 
#     of the archive 
#     Default values correspond to Mesodesma donacium mollusc shells
#       param[8]= Ny,       (integer) number of years spanned by 
#                           individual records
#       param[9] = N        (integer) number of shells per sample
#       param[10] = gap     (integer) number of random 1-month growth gaps
#                           per year.
#       param[11] = Tls     biological superior temperature limit for 
#                           skeletal growth
#       param[12] = Tli	    biological inferior temperature limit for 
#                           skeletal growth
#  param[13:24] = gb[1:12]  1 if the skeleton usually grows in month 
#                           1 (January) to 12 (December), 0 if it does not.
#
#    The following 4 parameters are related to the statistical gaussian noises
#       param[25] = sigma(w) standard deviation of weather monthly noise 
#                            (proxy unit)
#       param[26] = sigma(c) standard deviation of the effect of skeletal 
#                            micro-heterogeneity at the microsampling scale
#                            (diagenesis, vital effect...) (proxy unit)
#       param[27] = sigma(a) proxy analytical error (1 sigma) (proxy unit)
#       param[28] = sigma(s) spatial std deviation of annual mean(T) 
#                            in the studied area (C)
#  
#       param[29] = Niter    number of iteration of the Monte Carlo analysis
#
# Value:
#  A list of parameters computed form the MonteCarlo analysis
#     Tmn          Expected value of reconstructed annual mean temperature
#     EstTmn         Standard deviation of reconstructed annual mean teperature
#     vTmn         Exptected value of reconstructed variance of annual mean temperature
#     EstvTmn        Standard deviation of reconstructed variance of annual mean temperature
#     mdeltan      Exptected value of reconstructed mean annual amplitude
#     Estmdeltan     Standard deviation of reconstructed mean annual amplitude
#     vdeltan      Exptected value of reconstructed variance of the annual amplitude
#     Estvdeltan     Standard deviation of reconstructed variance of the annual amplitude
#     EsystTm      Systematic error for annual mean temperature
#     EsystvTm     Systematic error for variance of annual mean temperature
#     Esystdelta   Systematic error for mean annual amplitude
#     Esystvdelta  Systematic error for variance of mean annual amplitude
#     Ealpha       95% confidence interval for alpha
#     Ebeta        95% confidence interval for beta
#     RETm         95% confidence interval for annual mean temperature
#     REvTm        95% confidence interval for the variance of annual mean temperature
#     REmdelta     95% confidence interval for mean annual amplitude
#     REvdelta     95% confidence interval for the variance of mean annual amplitude
#
# References
# Carr, M., Sachs, J.P., Wallace, J.M., Exploring errors in paleoclimate 
# proxy reconstructions: I. Paleotemperature from molluscs and corals 
# geochemistry. Climate of the Past, submitted.
# matthieu.carre@univ-montp2.fr
#
# Examples
#
#  # Load Moco function 
#  source("MoCo.r")
#
#  # Computing error estimates wich Puerto Chicama 1925-2002 time series and 
#  # default parameters
#  moco_est = MoCo("PuertoChicama_19252002_nogap.txt","MoCo_param.txt",verbose = F)
#
#  # Producing fig.4 of Carre et al. article with SST Nino 1+2 1950-2009 time series
#  param = read.table("MoCo_param.txt")
#  param = param[,1]
#  param[25] = 0
#  param[26] = 0
#  compil_est = c()
#  seq_smn = seq(0,0.5,by=0.01)
#  for (sigmam in seq_smn){
#    param[27] = sigmam
#    moco_est = MoCo("SST-Nino1+2_1950-2009.txt",param = param,verbose = F)
#    compil_est=cbind(compil_est,unlist(moco_est))
#  }
#  par(mfrow = c(2,2))
#  plot(seq_smn,compil_est["Emdeltan",],type='l',xlab = "",ylab = expression(Delta*T))
#  plot(seq_smn,compil_est["Esystdelta",],type='l',xlab = "",ylab="")
#  plot(seq_smn,compil_est["Estvdelta",],type='l',xlab = expression(sigma[m]),ylab = expression(Var(Delta*T)))
#  plot(seq_smn,compil_est["Esystvdelta",],type='l',xlab = expression(sigma[m]),ylab="")



# Parameters passed as a file name or as a vector
if (file_param != ""){
  param = read.table(file_param)
  param = param[,1]
}
if(length(param) != 29)
  stop("param must be a vector of 29 elements")

# Reading monthly climatic time series from file
a=read.table(file_climate);
a = a[,ncol(a)]

# Parameters of the proxy calibration
alpha=param[1];  
beta=param[2];
sigmaT=param[3];
sigmaP=param[4];
r=param[5];
ncalib=param[6];
if (ncalib <3)
    stop("The calibration dataset must have more than 3 data points, (param[6]>3)");
df=ncalib-2;            # number of degrees of freedom
T0=param[7];

# Parameters of the biological archive
Ny=param[8];
N=param[9];
gap=param[10];
Tls=param[11];
if(Tls < max(a))
  stop("Tls < minimal climatic data value")
Tli=param[12];
if(Tli > max(a))
  stop("Tli > maximal climatic data value")

gb=param[13:24];

# Noise parameters
sigmaw=param[25];
sigmac=param[26];
sigmaa=param[27];
sigmam=sqrt(sigmaw^2+sigmac^2+sigmaa^2)*abs(alpha);
sigmas=param[28];

# Number of iterations
niter=param[29];

# number of complete years in the climatic monthly time series
nyrs = floor(length(a)/12)
# last year is deleted if uncomplete
a = a[1:(12*nyrs)]

# transormation of the vector in a matrix with 12 lines and nyrs columns
b = matrix(a, 12, nyrs); 

# Moyennes annuelles
c= colMeans(b)
yrs = which(is.nan(c))
if (length(yrs>0))
  stop(paste("these years have missing data : ",yrs))

# Calculating timeseries variables
Tm = mean(a);           # annual mean 
vTm=var(c);            # variance of the annual mean 
delta=apply(b,2,max)-apply(b,2,min); # annual ranges
mdelta=mean(delta);     # mean annual amplitude
vdelta=var(delta);      # variance of the annual amplitude
Tmax=max(a);            # maximum value
Tmin=min(a);            # minimum value

#______________________________________
# 
# Simulation
#______________________________________

# define and initialize variables
Tmi=c(); vTmi=c(); deltai=c(); mdeltai=c(); vdeltai=c();
Tmn=c(); ETmn=c(); vTmn=c(); EvTmn=c(); mdeltan=c(); Emdeltan=c(); vdeltan=c(); Evdeltan=c();

# dismiss months of systematic growth break
ts=b[which(gb==1),];
nmonth=dim(b)[1];

# dismiss random yearly growth breaks
if (gap >0 & gap<=nmonth){
  for (i in 1:nyrs)
     XG[sample(nmonth,gap),i]=0;
  nmonth = nmonth - gap
  ts=matrix(ts[which(XG==1)], nmonth, nyrs);
} else if (gap > nmonth){
   stop("The total number of random and systematic growth breaks is larger than 12 months");
}

# dismiss values beyond biological limits for skeletal growth

ts[which(ts>Tls | ts<Tli)] = NaN

#____________________________________
#
# Monte Carlo analysis
#_____________________________________
    
for (iter in 1: niter){
       
# random sampling of N windows of Ny years
   ts3=c();
   for (i in 1:N){
     sample = c(NaN)
     count = 0;
# A sample must have non NaN values
     while((length(which(!is.nan(sample))) == 0) & count < 1000){
       x=floor(runif(1,0,nyrs-Ny))*nmonth+1;  
       sample = ts[x:(x+nmonth*Ny-1)]
       count = count + 1
     }
# If impossible to find a sample with non NaN values, stop
     if (count == 1000)
       stop("Impossible to produce a sample with specified parameters")

     ts3 = rbind(ts3,sample) 
   }

# Add random normally distributed noises
g = rnorm(N,0,sigmas);  # spatial variability
h = rnorm(N*Ny*nmonth,0,sigmam); # monthly variability + analytical error
ts3 = ts3 + matrix(rep(g,Ny*nmonth),N,Ny*nmonth) + matrix(h,N,Ny*nmonth);

# calculate reconstructed variables from the simulated sample #iter
ts3=matrix(t(ts3), nmonth, N*Ny); 


Tmi = c(Tmi,mean(ts3,na.rm = T));   # annual mean #iter 
M=apply(ts3,2,mean,na.rm=T);
vTmi = c(vTmi, (N*Ny)/(N*Ny-1)*var(M,na.rm=T));    # variance of annual mean 
deltai=apply(ts3,2,max,na.rm=T)-apply(ts3,2,min,na.rm=T);
deltai[!is.finite(deltai)] = NaN
mdeltai=c(mdeltai,mean(deltai,na.rm=T)); # average annual amplitude
vdeltai=c(vdeltai,(N*Ny)/(N*Ny-1)*var(deltai,na.rm=T));  # variance of annual amplitude

} # end for iter

Tmn = mean(Tmi);            # Expected value of reconstructed annual mean
ETmn = sd(Tmi);            
vTmn=mean(vTmi);            # Exptected value of reconstructed variance of annual mean
EvTmn=sd(vTmi);
mdeltan = mean(mdeltai);    # Exptected value of reconstructed mean annual amplitude
Emdeltan = sd(mdeltai);
vdeltan = mean(vdeltai);    # Exptected value of reconstructed variance of the annual amplitude
Evdeltan = sd(vdeltai);
   
#Systematic Errors for Tm, var(Tm), delta(T), and var(delta(T))
EsystTm = Tmn - Tm
EsystvTm = vTmn - vTm
Esystdelta = mdeltan - mdelta
Esystvdelta = vdeltan - vdelta

#Standard Errors (1 sigma) for Tm, var(Tm), delta(T), and var(delta(T)
EstTm = ETmn
EstvTm = EvTmn
Estmdelta = Emdeltan
Estvdelta = Evdeltan

#Errors from the regression model at 95# confidence level for Tm, var(Tm), delta(T), and var(delta(T)
t=qt(0.025,df,lower.tail=F)                        # 0.025 quantile from Student distribution
Ealpha=t*(sigmaT/sigmaP)*sqrt((1-r^2)/(ncalib-2));  # 95% confidence interval for alpha
Ebeta=t*sigmaT*sqrt((1-r^2)/(ncalib-2));            # 95% confidence interval for beta
RETm = sqrt(Ebeta^2 + ((Tm-T0)^2)*Ealpha^2) 
REvTm = vTm*Ealpha/abs(alpha)
REmdelta = mdelta*Ealpha
REvdelta = vdelta*Ealpha/abs(alpha)


if(verbose){    
cat(' *************\n',
  'Systematic Errors for Tm, var(Tm), delta(T), and var(delta(T)) \n',
  EsystTm,'\t',EsystvTm,'\t',Esystdelta,'\t',Esystvdelta,'\n',
  '*************\n',
  'Standard Errors (1 sigma) for Tm, var(Tm), delta(T), and var(delta(T) \n',
  EstTm,'\t',EstvTm,'\t',Estmdelta,'\t',Estvdelta,'\n',
  '*************\n',
  'Errors from the regression model at 95% confidence level for Tm, var(Tm), delta(T), and var(delta(T)','\n',
  Ealpha,'\t', Ebeta,'\t',RETm,'\t',REvTm,'\t',REmdelta,'\t',REvdelta,'\n')

}

list(Tmn = Tmn, EstTm = EstTm, vTmn=vTmn, EstvTm=EstvTm, mdeltan = mdeltan, Estmdelta = Estmdelta, vdeltan = vdeltan, Estvdelta = Estvdelta, EsystTm = EsystTm, EsystvTm = EsystvTm, Esystdelta = Esystdelta, Esystvdelta = Esystvdelta, Ealpha=Ealpha, Ebeta= Ebeta, RETm = RETm, REvTm = REvTm, REmdelta = REmdelta, REvdelta = REvdelta)
}


