% MoCo program (Mollucs and Corals)
% Estimate the systematic and standard errors of paleoclimate statistics in reconstructions based on mollusk or corals geochemical proxies
%

q=input('please load "MoCo_timeseries.txt" as a 1 column txt file into MATLAB current directory, and press ENTER');
a=load('MoCo_timeseries.txt');
a=a(:,1);
% checking whether years are complete in the timeseries
nyrs=[]; nyrs=size(a,1)/12; nyrs=floor(nyrs);
a=a(1:12*nyrs,:);                               % last year is deleted if not complete
b=[]; c=[]; d=[]; b = reshape(a, 12, nyrs); c=mean(b); d=mean(c');
yrs=[]; yrs=find(isnan(c));
if d==NaN
    yrs
    error('these years have missing data')
end

% Calculating timeseries variables
Tm = []; vTm=[]; mdelta=[]; vdelta=[]; Tmax=[]; Tmin=[];
Tm = mean(a);           % annual mean 
vTm=var(c');            % variance of the annual mean 
delta=range(b); delta=delta';
mdelta=mean(delta);     % mean annual amplitude
vdelta=var(delta);      % variance of the annual amplitude
Tmax=max(a);            % maximum value
Tmin=min(a);            % minimum value

% Parameters of the proxy calibration
alpha=param(1);
beta=param(2);
sigmaT=param(3);
sigmaP=param(4);
r=param(5);
ncalib=param(6);
if ncalib <3
    error('The calibration dataset must have more than 3 data points, (param(6)>3)');
end
df=ncalib-2;            % number of degrees of freedom
T0=param(7);
% Parameters of the biological archive
Ny=param(8);
N=param(9);
gap=param(10);
Tls=param(11);
Tli=param(12);
gb=param(13:24);
% Noises parameters
sigmaw=param(25);
sigmac=param(26);
sigmaa=param(27);
sigmam=sqrt(sigmaw^2+sigmac^2+sigmaa^2)*abs(alpha);
sigmas=param(28);

niter=param(29);

%______________________________________
% 
% Simulation
%______________________________________

% define and initialize variables
Tmi=[]; vTmi=[]; deltai=[]; mdeltai=[]; vdeltai=[];
Tmn=[]; ETmn=[]; vTmn=[]; EvTmn=[]; mdeltan=[]; Emdeltan=[]; vdeltan=[]; Evdeltan=[];

% dismiss months of systematic growth break
xgb=[]; xgb=find(gb==1);
ts=b(xgb, :);
nmonth=size(xgb,2);

% dismiss random yearly growth breaks
if gap<=nmonth
    GP=[]; 
    for i=1:nyrs
        GP(:,i)=randperm(nmonth)';
    end
    xgap=[]; xgap=GP(1:gap, :);
    XG=ones(12, nyrs);
    for i=1:nyrs
         XG(xgap(:,i),i)=0;
    end
    XYG=find(XG==1);
    ts=ts(XYG); nmonth=nmonth-gap; ts=reshape(ts, nmonth, nyrs);
else
    error('The total number of random and systematic growth breaks is larger than 12 months');
end

% dismiss values beyond biological temperature thresholds for skeletal growth

ts=reshape(ts, nmonth*nyrs, 1);
for i=1:size(ts)
    if ts(i,1)>Tls
        ts(i,1)=NaN;
    elseif ts(i,1)<Tli
        ts(i,1)=NaN;
    else
    end
end
if length(find(~isnan(ts)))<1
    error('The timeseries values are beyond biological limits'); 
end
ts=reshape(ts, nmonth, nyrs); ts=ts';

%____________________________________
%
% Monte Carlo analysis
%_____________________________________
    

for iter = 1: niter
   x=[]; y=[]; ts2=[]; ts3=[]; M=[];
   g=[]; h=[]; 
       
% random sampling of N windows of Ny years
for i=1:N
x=randi([1, nyrs-Ny]);
ts2 = ts(x:x+Ny-1, :); ts2=ts2'; ts2 = reshape(ts2, 1, Ny*nmonth); 
ts3(i,:)=ts2;
end

% Add random normally distributed noises
g = randn(N,1)*sigmas;  % spatial variability
h = randn(N,Ny*nmonth)*sigmam; % monthly variability + analytical error
ts3 = ts3 + g*ones(1,Ny*nmonth) + h;

% calculate reconstructed variables from the simulated sample #iter
ts3=ts3'; ts3 = reshape(ts3, N*Ny*nmonth, 1);
ts3 = reshape(ts3, nmonth, N*Ny); 
y=find(~isnan(ts3));
Tmi(iter,1)=mean(ts3(y));   % annual mean #iter 
for j=1:N*Ny
    xj=[]; tsj=[]; tsj=ts3(:,j); xj=find(~isnan(tsj));
    M(j,1)=mean(tsj(xj));
end
x2=[]; x2=find(~isnan(M)); 
vTmi(iter,1)=var(M(x2));    % variance of annual mean 
deltai=range(ts3);
x3=[]; x3=find(~isnan(deltai));
mdeltai(iter,1)=mean(deltai(x3)); % average annual amplitude
vdeltai(iter,1)=var(deltai(x3));  % variance of annual amplitude
end

Tmn = mean(Tmi);            % Esperance of reconstructed annual mean
ETmn = std(Tmi);            
vTmn=mean(vTmi)*N*Ny/(N*Ny-1);            % Esperance of reconstructed variance of annual mean
EvTmn=std(vTmi)*N*Ny/(N*Ny-1);
mdeltan = mean(mdeltai);    % esperance of reconstructed mean annual amplitude
Emdeltan = std(mdeltai);
vdeltan = mean(vdeltai)*N*Ny/(N*Ny-1);    % esperance of reconstructed variance of the annual amplitude
Evdeltan = std(vdeltai)*N*Ny/(N*Ny-1);
      
%__________________________________________
%
disp('Systematic Errors for Tm, var(Tm), delta(T), and var(delta(T))')
%___________________________________________


EsystTm = Tmn - Tm
EsystvTm = vTmn - vTm
Esystdelta = mdeltan - mdelta
Esystvdelta = vdeltan - vdelta

%_____________________________________________
%
disp('Standard Errors (1 sigma) for Tm, var(Tm), delta(T), and var(delta(T)')
%______________________________________________

EstTm = ETmn
EstvTm = EvTmn
Estmdelta = Emdeltan
Estvdelta = Evdeltan

%_____________________________________________
%
disp('Errors from the regression model at 95% confidence level for Tm, var(Tm), delta(T), and var(delta(T)')
%______________________________________________

if ncalib<=102
    t=Student95(df);  
else
    t=1.98;   
end                                                 % student variable value T so P(abs(t)>T)=0,05
Ealpha=t*(sigmaT/sigmaP)*sqrt((1-r^2)/(ncalib-2));  % 95% confidence interval for alpha
Ebeta=t*sigmaT*sqrt((1-r^2)/(ncalib-2));            % 95% confidence interval for beta
RETm = sqrt(Ebeta^2 + ((Tm-T0)^2)*Ealpha^2) 
REvTm = vTm*Ealpha/abs(alpha)
REmdelta = mdelta*Ealpha
REvdelta = vdelta*Ealpha/abs(alpha)





