function [age,varargout]=Nye_interp(z,ACP,H,varargin)
% Interpolates ice-ages at depths z, given the Age Control Points (ACP) and core
% depth (H)
% [age,varargout]=Nye_interp(z,ACP,H,...)
% 
% Inputs
%     z: depths to interpolate ages for (this must be ice eq depth)
%     ACP: N x 2 matrix of Age Control Points ([Depths, Age])
%         *Depths must be in ice equivalent depths
%         *Age is assumed to be Ice Age (use 'delta_age' input for Gas Age)
%     H: Ice Sheet Thickness under strain conditions (also, ice eq depth)
% 
% Varargin
%     .delta_age: Use if ACPs are input as gas ages --> argument out becomes gas age.
%     .ref_date: Define 'Present' (default = 2000 CE)
%     .date0: Age of ice at 0 m depth (date core drilled?)
% 
% Outputs
%     age: age of ice at depth 'z'
%     b_dot: accumulation history (ice age);
%            b_dot is vector of length(ACP)+1;
%            b_dot(1) represents accumulation between 0-ACP(1) years;
%            b_dot(end) represents accumulation between ACP(end)-bottom of
%               core
%     ALT: annual layer thickness at depth 'z';
% 
% 
% Written by James E. Lee, College of Earth, Ocean, Atmospheric Sciences,
% Oregon State University
% Last Update: 03/16/15

%% Deal with input format
if ~isnumeric(z) || ~isnumeric(ACP) || ~isnumeric(H)
    error('Missing required input (z,ACP,H)')
end
  
%% Formating varargin fields
if rem(size(varargin,2),2)==0
    fields=reshape(varargin,2,size(varargin,2)/2);
    fields=cell2struct(fields(2,:),fields(1,:),2);
else
    fprintf('Input fields must enter as pairs')
end

%% Preset Values for optional inputs
delta_age=0;
ref_date=2000; %sets reference date
date0=2012; %age of depth at 0 depth

if isfield(fields,'delta_age');delta_age=fields.delta_age;end
if isfield(fields,'ref_date');ref_date=fields.ref_date;end
if isfield(fields,'date0');date0=fields.date0;end

age=nan(size(z));
%% Deal with z formatting
if ~isvector(z)
    error('Error: z must be vector')
end

nan_ind=isnan(z) | z>=H;
if max(z)>=H;
    fprintf('Input z abbreviated to depths less than H.\n');
    z=z(z<H);
end
z=reshape(z,[],1); %Reshapes z into column vector;

%% Deal with ACP formatting
%convert to ice age if ACP is in Gas Age
ACP(:,2)=ACP(:,2)+delta_age; 
%add ACP at 0 m depth
if ACP(1,1); 
    ACP=[0 ref_date-date0;ACP];
end

%sort acp's
id=isnan(sum(ACP,2));ACP=ACP(~id,:);
if ~issorted(ACP(:,1));ACP=sortrows(ACP,1);end
if ~issorted(ACP(:,2));error('Error: ACP not in stratigraphic order');end

Z=ACP(:,1); %Age control depths
Age=ACP(:,2); %Age control ages

%% Calculate stuff
% mean accumulation rate between ACP's;
b_dot=H*log((H-Z(1:end-1))./(H-Z(2:end)))./diff(Age);
%Assume accumulation rate remains constant deeper than last ACP
b_dot(end+1)=b_dot(end); 

% solve for ages
if numel(z)<25000;
    X=z*ones(size(Z')); %expanded vector of z
    Y=ones(size(z))*Z'; %expanded vector of Z
    j=sum(X>=Y,2); %Index of ACP's shallower/younger than elements in z
    clear X Y
else
    j=nan(size(z));
    for i=1:numel(z);
        j(i)=sum(Z<=z(i));
    end
end

%Interpolate Ice Age
out1=Age(j)+(H./b_dot(j)).*log((H-Z(j))./(H-z)); 
%Interpolate Gas Age
out1=out1-delta_age;

%Define output: 'age'
age(~nan_ind)=out1;

%% Varargout
%First Optional Argout : accumulation rate b/w tie points
varargout={b_dot};
%Second Optional Argout : Annual Layer Thickness at z
varargout{2}=b_dot(j).*(H-z)/H;