function [z,varargout]=NyeInv_interp(age,ACP,H,varargin)
% Interplates ice-ages at depths z, given the Age Control Points (ACP) and core
% depth (H)
% 
% Inputs
%     age: age to interpolate depth for
%     ACP: N x 2 matrix of Age Control Points ([Depths, Age])
%     H: Full Length of core under strain conditions
% 
% Varargin
%     .delta_age: if this argument is used, 'age' and 'ACP's are input as
%         gas ages, 'z' is the depth at which that gas age occurs
%     .ref_date: Define 'Present' (default = 2000 CE)
%     .date0: Age of ice at 0 m depth (date core drilled?)
% 
% 
% Outputs
%     age: age of ice at depth 'z'
%     b_dot: accumulation history (ice age);
% 
% 
% Written by James E. Lee, College of Earth, Ocean, Atmospheric Sciences,
% Oregon State University
% Last Update: 03/16/15

%% Deal with input format
if ~isnumeric(age) || ~isnumeric(ACP) || ~isnumeric(H)
    error('Missing required input (age,ACP,H)')
end

%% Formating varargin fields
if rem(size(varargin,2),2)==0
    fields=reshape(varargin,2,size(varargin,2)/2);
    fields=cell2struct(fields(2,:),fields(1,:),2);
else
    fprintf('Input fields must enter as pairs')
end

delta_age=0;
if isfield(fields,'delta_age');delta_age=fields.delta_age;end

%% Preset Values for optional inputs
delta_age=0;
ref_date=2000; %sets reference date
date0=2012; %age of depth at 0 depth

if isfield(fields,'delta_age');delta_age=fields.delta_age;end
if isfield(fields,'ref_date');ref_date=fields.ref_date;end
if isfield(fields,'date0');date0=fields.date0;end
  
%% Deal with z formatting
if ~isvector(age)
    error('Error: age must be vector')
end
% if max(age)>max(ACP(:,2));
%     fprintf('Input z abbreviated to depths less than H');
%     age=age(age<max(ACP(:,2)));
% end
age=reshape(age,[],1)+delta_age;

%% Deal with ACP formatting

ACP(:,2)=ACP(:,2)+delta_age; %convert to ice age from ACP
if ACP(1,1); %adds ACP at 0 m depth
    ACP=[0 ref_date-date0;ACP];
end

id=isnan(sum(ACP,2));ACP=ACP(~id,:);
if ~issorted(ACP(:,1));ACP=sortrows(ACP,1);end
if ~issorted(ACP(:,2));error('Error: ACP not in stratigraphic order');end

Z=ACP(:,1); %Age control depths
Age=ACP(:,2); %Age control ages

%% Calculate stuff

%Method 2;
b_dot=H*log((H-Z(1:end-1))./(H-Z(2:end)))./diff(Age);
b_dot(end+1)=b_dot(end);

% solve for depths
j=sum(age*ones(size(Age'))>ones(size(age))*Age',2);
z=H-(H-Z(j)).*exp(-(b_dot(j).*(age-Age(j)))/H);
% j=sum(z*ones(size(Z'))>ones(size(z))*Z',2); %Index of ACP's shallower/younger than elements in z
% age=Age(j)+(H./b_dot(j)).*log((H-Z(j))./(H-z));

%% Varargout
varargout(1)={b_dot};