# Spectral analysis of the GR series
library(multitaper)
library(dplR)
library(astrochron)

# Set working directory under Mac OS X - to adapt
setwd ("~/Desktop/TestUncertainty/Manuscript/Revisions/Final files")

# Set working directory under Windows - to adapt
setwd ("D://Docs boulot/TestUncertainty/Manuscript/Revisions/Final files")

# 01 - Preparing data ----
frq   <- seq(from=0.02, to=0.48, by=0.02)
tim   <- seq(from=1, to=600, by=1)
dt    <- 1
sig   <- 0
for (i in 1:length(frq)) {
  print(frq[i])
  sig <- sig+sin(2*pi*frq[i]*tim)
}

SpecSin <- redfit(sig,tim, tType="time", nsim=1000, mctest=TRUE, rhopre=0,
                  ofac=1,hifac=1, n50=1, iwin=0, txOrdered = TRUE)
freq    <- SpecSin$freq
PowR    <- SpecSin$gxx


# 02 - Randomization gamma ----
deltaT = tim[2:length(tim)]-tim[1:length(tim)-1]
nsimul = 10 # Number of simulations: to adapt
d1     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d2     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d3     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
for (i in 1:length(deltaT)) {
  ave = deltaT[i]
  s1  = 0.05*ave
  s2  = 0.10*ave
  s3  = 0.15*ave
  
  v1  = s1^2
  v2  = s2^2
  v3  = s3^2
  
  sc1 = v1/ave
  sc2 = v2/ave
  sc3 = v3/ave
  
  sh1 = ave/sc1
  sh2 = ave/sc2
  sh3 = ave/sc3
  
  d1[i,1:nsimul] = rgamma(n=nsimul,shape=sh1,scale=sc1)
  d2[i,1:nsimul] = rgamma(n=nsimul,shape=sh2,scale=sc2)
  d3[i,1:nsimul] = rgamma(n=nsimul,shape=sh3,scale=sc3)
}

tr1 = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr2 = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr3 = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr1[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr2[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr3[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)

for (j in 1:nsimul) {
  for (i in 1:length(deltaT)) {
    tr1[i+1,j] = tr1[i,j]+d1[i,j]
    tr2[i+1,j] = tr2[i,j]+d2[i,j]
    tr3[i+1,j] = tr3[i,j]+d3[i,j]
  }
}


# 04 - 2pi-MTM spectrum of the time-randomized series ----
loop    = 0
nfreq   = length(freq)

freq_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
freq_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
freq_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

PowR_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
PowR_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
PowR_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

lres1   = matrix(rep(0,nsimul),1,nsimul)
lres2   = matrix(rep(0,nsimul),1,nsimul)
lres3   = matrix(rep(0,nsimul),1,nsimul)

for (i in 1:nsimul) {
  s        = Sys.time()
  loop     = loop+1
  print(loop)
  trand1   = tr1[,i]
  trand2   = tr2[,i]
  trand3   = tr3[,i]
  
  SpecLCH_R1  = redfit(sig,trand1, tType="time", nsim=1000, mctest=TRUE, rhopre=0,
                       ofac=1, hifac=1, n50=1, iwin=0, txOrdered = TRUE)
  freq_R1[1:length(SpecLCH_R1$freq),i] = SpecLCH_R1$freq
  PowR_R1[1:length(SpecLCH_R1$freq),i] = SpecLCH_R1$gxx
  
  SpecLCH_R2  = redfit(sig,trand2, tType="time", nsim=1000, mctest=TRUE, rhopre=0,
                       ofac=1, hifac=1, n50=1, iwin=0, txOrdered = TRUE)
  freq_R2[1:length(SpecLCH_R2$freq),i] = SpecLCH_R2$freq
  PowR_R2[1:length(SpecLCH_R2$freq),i] = SpecLCH_R2$gxx
  
  SpecLCH_R3  = redfit(sig,trand3, tType="time", nsim=1000, mctest=TRUE, rhopre=0,
                       ofac=1, hifac=1, n50=1, iwin=0, txOrdered = TRUE)
  freq_R3[1:length(SpecLCH_R3$freq),i] = SpecLCH_R3$freq
  PowR_R3[1:length(SpecLCH_R3$freq),i] = SpecLCH_R3$gxx
  
  lres1[i]   = length(SpecLCH_R1$freq)
  lres2[i]   = length(SpecLCH_R2$freq)
  lres3[i]   = length(SpecLCH_R3$freq)
  a           = Sys.time()
  TimElaps    = a-s
  print(TimElaps)
}


# 05 - Calculating the average spectrum and relative error ----

FreqMean1 = matrix(rep(0,min(lres1))  ,nrow=min(lres1),ncol=1)
PrawMean1 = matrix(rep(0,min(lres1))  ,nrow=min(lres1),ncol=1)
powR1_p95 = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)

FreqMean2 = matrix(rep(0,min(lres2))  ,nrow=min(lres2),ncol=1)
PrawMean2 = matrix(rep(0,min(lres2))  ,nrow=min(lres2),ncol=1)
powR2_p95 = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)

FreqMean3 = matrix(rep(0,min(lres3))  ,nrow=min(lres3),ncol=1)
PrawMean3 = matrix(rep(0,min(lres3))  ,nrow=min(lres3),ncol=1)
powR3_p95 = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)

for (i in 1:min(lres1)) {
  FreqMean1[i]  = mean(freq_R1[i,])
  PrawMean1[i]  = mean(PowR_R1[i,])
  powR1_p95[i,] = quantile(PowR_R1[i,], c(.05, .95))
  
}

for (i in 1:min(lres2)) {
  FreqMean2[i]  = mean(freq_R2[i,])
  PrawMean2[i]  = mean(PowR_R2[i,])
  powR2_p95[i,] = quantile(PowR_R2[i,], c(.05, .95))
}

for (i in 1:min(lres3)) {
  FreqMean3[i] <- mean(freq_R3[i,])
  PrawMean3[i] <- mean(PowR_R3[i,])
  powR3_p95[i,] = quantile(PowR_R3[i,], c(.05, .95))
}

indfmax        <- matrix(rep(0,length(frq)),nrow=length(frq),ncol=1)
for (i in 1:length(frq)) {
  diff         <- abs(freq-frq[i])
  mdiff        <- min(diff)
  indfmax[i]   <- match(mdiff,diff)
}

ErrRelRaw1     <- 100*abs(PrawMean1[indfmax]-PowR[indfmax])/PowR[indfmax]
ErrRelRaw2     <- 100*abs(PrawMean2[indfmax]-PowR[indfmax])/PowR[indfmax]
ErrRelRaw3     <- 100*abs(PrawMean3[indfmax]-PowR[indfmax])/PowR[indfmax]

IndFrst1Raw    <- min(which(ErrRelRaw1>10))
IndFrst2Raw    <- min(which(ErrRelRaw2>10))
IndFrst3Raw    <- min(which(ErrRelRaw3>10))

FirstFr1Raw    <- freq[indfmax[IndFrst1Raw]]
FirstFr2Raw    <- freq[indfmax[IndFrst2Raw]]
FirstFr3Raw    <- freq[indfmax[IndFrst3Raw]]


# 06 - Graphics spectra ----

par(mfrow=c(1,4))
plot (freq,PowR, type="l", col="black",xlab="Frequency (cycles/m)",
      ylab="Power",ylim=c(0,325))
title(main="Lomb-Scargle raw and red-noise corrected spectra")

plot (FreqMean1,powR1_p95[,1],col="lightgrey",type="l", 
      xlab="Frequency (cycles/m)",ylab="Power",ylim=c(0,325))
lines(FreqMean1,powR1_p95[,2],col="lightgrey")
lines(FreqMean1,PrawMean1,col="darkorange")
title("Lomb-Scargle spectrum - 5% uncertainty on the sample position")

plot (FreqMean2,powR2_p95[,1],col="lightgrey",type="l",
      xlab="Frequency (cycles/m)", ylab="Power",ylim=c(0,325))
lines(FreqMean2,powR2_p95[,2],col="lightgrey")
lines(FreqMean2,PrawMean2,col="red")
title("Lomb-Scargle spectrum - 10% uncertainty on the sample position")

plot (FreqMean3,powR3_p95[,1],col="lightgrey",type="l",
      xlab="Frequency (cycles/m)", ylab="Power",ylim=c(0,325))
lines(FreqMean3,powR3_p95[,2],col="lightgrey")
lines(FreqMean3,PrawMean3,col="darkorange4")
title("Lomb-Scargle spectrum - 15% uncertainty on the sample position")

par  (mfrow=c(1,1))
plot (frq,ErrRelRaw1,type="l", col="darkorange", xlim=c(0,0.5), ylim=c(0,100), 
      xlab="Frequency (cycles/m)", ylab="Relative change in power (%)")
lines(frq,ErrRelRaw2, col="red", xlim=c(0,0.5), ylim=c(0,100), xlab="Frequency (cycles/m)", 
      ylab="Relative change in power (%)")
lines(frq,ErrRelRaw3, col="darkorange4", xlim=c(0,0.5), ylim=c(0,100), xlab="Frequency (cycles/m)", 
      ylab="Relative change in power (%)")
lines(c(0,0.5),c(10,10),lty="dashed",col="black")
lines(c(FirstFr1Raw,FirstFr1Raw),c(0,100),lty="dashed",col="darkorange")
lines(c(FirstFr2Raw,FirstFr2Raw),c(0,100),lty="dashed",col="red")
lines(c(FirstFr3Raw,FirstFr3Raw),c(0,100),lty="dashed",col="darkorange4")
title("Relative change in power (%)")


propFr1Raw <- FirstFr1Raw/max(freq)*100
propFr2Raw <- FirstFr2Raw/max(freq)*100
propFr3Raw <- FirstFr3Raw/max(freq)*100

print(propFr1Raw)
print(propFr2Raw)
print(propFr3Raw)


