# Spectral analysis of the MS series
library(multitaper)
library(dplR)
library(astrochron)

# Set working directory under Mac OS X - to adapt
setwd  ("~/Desktop/TestUncertainty/Manuscript/Revisions/Final files")

# Set working directory under Windows - to adapt
setwd  ("D://Docs boulot/TestUncertainty/Manuscript/Revisions/Final files")

DetrendLaThure <- function(datRaw) {
  dat1  = datRaw[1:333,]
  dat2  = datRaw[334:371,]
  dat3  = datRaw[372:length(dat[,1]),]
  
  ModL1 = lm(dat1[,2]~dat1[,1])
  ModL2 = lm(dat2[,2]~dat2[,1])
  ModL3 = lm(dat3[,2]~dat3[,1])
  
  lin1  = predict(ModL1,data.frame(x = dat1[,1]))
  lin2  = predict(ModL2,data.frame(x = dat2[,1]))
  lin3  = predict(ModL3,data.frame(x = dat3[,1]))
  linT  = matrix(rep(0,length(dat[,1])),nrow=length(dat[,1]),ncol=1)
  linT[1:333]   = lin1
  linT[334:371] = lin2
  linT[372:length(datRaw[,1])] = lin3
  
  sD1   = dat1[,2]-lin1
  sD2   = dat2[,2]-lin2
  sD3   = dat3[,2]-lin3
  
  sD          = matrix(rep(0,length(datRaw[,1])),nrow=length(datRaw[,1]),ncol=1)
  sD[1:333]   = sD1
  sD[334:371] = sD2
  sD[372:length(datRaw[,1])] = sD3
  datD        = data.frame(datRaw[,1],sD)
  
  datD[,1]     = round(datD[,1],3)
  datID        = linterp(datD,dt=0.001,genplot=F,verbose=F)
  iamp         = hilbert(datID, padfac=1, demean=T,detrend=F,output=T,addmean=T,genplot=F,verbose=F)
  iamp[,1]     = round(iamp[,1],3)
  IndSameLevel = match(datD[,1],iamp[,1])
  iampSameLev  = iamp[IndSameLevel,]
  RegAmp       = lowess(iampSameLev,f=0.1)
  sigAD        = datD[,2]/RegAmp$y
  sigADN       = (sigAD-mean(sigAD))/sd(sigAD)
  datDet       = data.frame(datD[,1],sD,sigADN)
  detLT=list(trend=linT,datDet=datDet,InstAmp=iampSameLev,TrendAmp=RegAmp)
  return(detLT)
}


# 01 - Preparing data ----
dat     = read.table("LaThure_MS_R.txt", header=T)
dtmoy   = 0.38

DetLT   = DetrendLaThure(dat)
level   = DetLT$datDet[,1]
trend   = DetLT$trend
sigD    = DetLT$datDet[,2]
sigDN   = DetLT$datDet[,3]
InstAmp = DetLT$InstAmp
TrAmp   = DetLT$TrendAmp

SpecLTH <- redfit(sigDN,level, tType="time", nsim=1000, mctest=TRUE, 
                  ofac=1,hifac=1, n50=1, p = c(0.10, 0.05, 0.01), iwin=0, txOrdered = TRUE)
freq    <- SpecLTH$freq
PowC    <- SpecLTH$gxxc
CL50    <- SpecLTH$gredth
CL90    <- SpecLTH$gredth * qchisq(0.90, df=2)/2
CL95    <- SpecLTH$gredth * qchisq(0.95, df=2)/2
CL99    <- SpecLTH$gredth * qchisq(0.99, df=2)/2


# 02 - Randomization gamma ----
tim    = dat[,1]
deltaT = tim[2:length(tim)]-tim[1:length(tim)-1]
nsimul = 10 # to adapt
d1     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d2     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d3     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
for (i in 1:length(deltaT)) {
  ave = deltaT[i]
  s1  = 0.05*ave
  s2  = 0.10*ave
  s3  = 0.15*ave
  
  v1  = s1^2
  v2  = s2^2
  v3  = s3^2
  
  sc1 = v1/ave
  sc2 = v2/ave
  sc3 = v3/ave
  
  sh1 = ave/sc1
  sh2 = ave/sc2
  sh3 = ave/sc3
  
  d1[i,1:nsimul] = rgamma(n=nsimul,shape=sh1,scale=sc1)
  d2[i,1:nsimul] = rgamma(n=nsimul,shape=sh2,scale=sc2)
  d3[i,1:nsimul] = rgamma(n=nsimul,shape=sh3,scale=sc3)
}

tr1 = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr2 = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr3 = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr1[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr2[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr3[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)

for (j in 1:nsimul) {
  for (i in 1:length(deltaT)) {
    tr1[i+1,j] = tr1[i,j]+d1[i,j]
    tr2[i+1,j] = tr2[i,j]+d2[i,j]
    tr3[i+1,j] = tr3[i,j]+d3[i,j]
  }
}


# 03 - 2pi-MTM spectrum of the time-randomized series ----
loop    = 0
sig     = dat[,2]
nfreq   = length(freq)

freq_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
freq_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
freq_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

PowC_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
PowC_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
PowC_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

CL50_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
CL50_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
CL50_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

CL90_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
CL90_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
CL90_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

CL95_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
CL95_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
CL95_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

CL99_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
CL99_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
CL99_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

lres1   = matrix(rep(0,nsimul),1,nsimul)
lres2   = matrix(rep(0,nsimul),1,nsimul)
lres3   = matrix(rep(0,nsimul),1,nsimul)

for (i in 1:nsimul) {
  s        = Sys.time()
  loop     = loop+1
  print(loop)
  trand1   = tr1[,i]
  trand2   = tr2[,i]
  trand3   = tr3[,i]
  
  datR1    = data.frame(trand1,sig)
  datR2    = data.frame(trand2,sig)
  datR3    = data.frame(trand3,sig)
  
  detLT_R1 = DetrendLaThure(datR1)
  detLT_R2 = DetrendLaThure(datR2)
  detLT_R3 = DetrendLaThure(datR3)
  
  SpecLTH_R1  = redfit(detLT_R1$datDet[,3],detLT_R1$datDet[,1], tType="time", nsim=1000, mctest=TRUE, 
                       ofac=1,hifac=1, n50=1, p = c(0.10, 0.05, 0.01), iwin=0, txOrdered = TRUE)
  freq_R1[1:length(SpecLTH_R1$freq),i] = SpecLTH_R1$freq
  PowC_R1[1:length(SpecLTH_R1$freq),i] = SpecLTH_R1$gxxc
  CL50_R1[1:length(SpecLTH_R1$freq),i] = SpecLTH_R1$gredth
  CL90_R1[1:length(SpecLTH_R1$freq),i] = SpecLTH_R1$gredth * qchisq(0.90, df=2)/2
  CL95_R1[1:length(SpecLTH_R1$freq),i] = SpecLTH_R1$gredth * qchisq(0.95, df=2)/2
  CL99_R1[1:length(SpecLTH_R1$freq),i] = SpecLTH_R1$gredth * qchisq(0.99, df=2)/2
  
  SpecLTH_R2  = redfit(detLT_R2$datDet[,3],detLT_R2$datDet[,1], tType="time", nsim=1000, mctest=TRUE, 
                       ofac=1,hifac=1, n50=1, p = c(0.10, 0.05, 0.01), iwin=0, txOrdered = TRUE)
  freq_R2[1:length(SpecLTH_R2$freq),i] = SpecLTH_R2$freq
  PowC_R2[1:length(SpecLTH_R2$freq),i] = SpecLTH_R2$gxxc
  CL50_R2[1:length(SpecLTH_R2$freq),i] = SpecLTH_R2$gredth
  CL90_R2[1:length(SpecLTH_R2$freq),i] = SpecLTH_R2$gredth * qchisq(0.90, df=2)/2
  CL95_R2[1:length(SpecLTH_R2$freq),i] = SpecLTH_R2$gredth * qchisq(0.95, df=2)/2
  CL99_R2[1:length(SpecLTH_R2$freq),i] = SpecLTH_R2$gredth * qchisq(0.99, df=2)/2
  
  SpecLTH_R3  = redfit(detLT_R3$datDet[,3],detLT_R3$datDet[,1], tType="time", nsim=1000, mctest=TRUE, 
                       ofac=1,hifac=1, n50=1, p = c(0.10, 0.05, 0.01), iwin=0, txOrdered = TRUE)
  freq_R3[1:length(SpecLTH_R3$freq),i] = SpecLTH_R3$freq
  PowC_R3[1:length(SpecLTH_R3$freq),i] = SpecLTH_R3$gxxc
  CL50_R3[1:length(SpecLTH_R3$freq),i] = SpecLTH_R3$gredth
  CL90_R3[1:length(SpecLTH_R3$freq),i] = SpecLTH_R3$gredth * qchisq(0.90, df=2)/2
  CL95_R3[1:length(SpecLTH_R3$freq),i] = SpecLTH_R3$gredth * qchisq(0.95, df=2)/2
  CL99_R3[1:length(SpecLTH_R3$freq),i] = SpecLTH_R3$gredth * qchisq(0.99, df=2)/2
  
  lres1[i]   = length(SpecLTH_R1$freq)
  lres2[i]   = length(SpecLTH_R2$freq)
  lres3[i]   = length(SpecLTH_R3$freq)
  a           = Sys.time()
  TimElaps    = a-s
  print(TimElaps)
}


# 04 - Calculating the average spectrum and relative error ----

FreqMean1 = matrix(rep(0,  min(lres1)),nrow=min(lres1),ncol=1)
PcorMean1 = matrix(rep(0,  min(lres1)),nrow=min(lres1),ncol=1)
CL50Mean1 = matrix(rep(0,  min(lres1)),nrow=min(lres1),ncol=1)
CL90Mean1 = matrix(rep(0,  min(lres1)),nrow=min(lres1),ncol=1)
CL95Mean1 = matrix(rep(0,  min(lres1)),nrow=min(lres1),ncol=1)
CL99Mean1 = matrix(rep(0,  min(lres1)),nrow=min(lres1),ncol=1)
powC1_p95 = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)
CL501_p95 = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)
CL901_p95 = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)
CL951_p95 = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)
CL991_p95 = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)

FreqMean2 = matrix(rep(0,  min(lres2)),nrow=min(lres2),ncol=1)
PcorMean2 = matrix(rep(0,  min(lres2)),nrow=min(lres2),ncol=1)
CL50Mean2 = matrix(rep(0,  min(lres2)),nrow=min(lres2),ncol=1)
CL90Mean2 = matrix(rep(0,  min(lres2)),nrow=min(lres2),ncol=1)
CL95Mean2 = matrix(rep(0,  min(lres2)),nrow=min(lres2),ncol=1)
CL99Mean2 = matrix(rep(0,  min(lres2)),nrow=min(lres2),ncol=1)
powC2_p95 = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)
CL502_p95 = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)
CL902_p95 = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)
CL952_p95 = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)
CL992_p95 = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)

FreqMean3 = matrix(rep(0,  min(lres3)),nrow=min(lres3),ncol=1)
PcorMean3 = matrix(rep(0,  min(lres3)),nrow=min(lres3),ncol=1)
CL50Mean3 = matrix(rep(0,  min(lres3)),nrow=min(lres2),ncol=1)
CL90Mean3 = matrix(rep(0,  min(lres3)),nrow=min(lres3),ncol=1)
CL95Mean3 = matrix(rep(0,  min(lres3)),nrow=min(lres3),ncol=1)
CL99Mean3 = matrix(rep(0,  min(lres3)),nrow=min(lres3),ncol=1)
powC3_p95 = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)
CL503_p95 = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)
CL903_p95 = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)
CL953_p95 = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)
CL993_p95 = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)

for (i in 1:min(lres1)) {
  FreqMean1[i]  = mean(freq_R1[i,])
  PcorMean1[i]  = mean(PowC_R1[i,])
  CL50Mean1[i]  = mean(CL50_R1[i,])
  CL90Mean1[i]  = mean(CL90_R1[i,])
  CL95Mean1[i]  = mean(CL95_R1[i,])
  CL99Mean1[i]  = mean(CL99_R1[i,])
  powC1_p95[i,] = quantile(PowC_R1[i,], c(.05, .95))
  CL501_p95[i,] = quantile(CL50_R1[i,], c(.05, .95))
  CL901_p95[i,] = quantile(CL90_R1[i,], c(.05, .95))
  CL951_p95[i,] = quantile(CL95_R1[i,], c(.05, .95))
  CL991_p95[i,] = quantile(CL99_R1[i,], c(.05, .95))  
}

for (i in 1:min(lres2)) {
  FreqMean2[i]  = mean(freq_R2[i,])
  PcorMean2[i]  = mean(PowC_R2[i,])
  CL50Mean2[i]  = mean(CL50_R2[i,])
  CL90Mean2[i]  = mean(CL90_R2[i,])
  CL95Mean2[i]  = mean(CL95_R2[i,])
  CL99Mean2[i]  = mean(CL99_R2[i,])
  powC2_p95[i,] = quantile(PowC_R2[i,], c(.05, .95))
  CL502_p95[i,] = quantile(CL50_R2[i,], c(.05, .95))
  CL902_p95[i,] = quantile(CL90_R2[i,], c(.05, .95))
  CL952_p95[i,] = quantile(CL95_R2[i,], c(.05, .95))
  CL992_p95[i,] = quantile(CL99_R2[i,], c(.05, .95))
}

for (i in 1:min(lres3)) {
  FreqMean3[i]  = mean(freq_R3[i,])
  PcorMean3[i]  = mean(PowC_R3[i,])
  CL50Mean3[i]  = mean(CL50_R3[i,])
  CL90Mean3[i]  = mean(CL90_R3[i,])
  CL95Mean3[i]  = mean(CL95_R3[i,])
  CL99Mean3[i]  = mean(CL99_R3[i,])
  powC3_p95[i,] = quantile(PowC_R3[i,], c(.05, .95))
  CL503_p95[i,] = quantile(CL50_R3[i,], c(.05, .95))
  CL903_p95[i,] = quantile(CL90_R3[i,], c(.05, .95))
  CL953_p95[i,] = quantile(CL95_R3[i,], c(.05, .95))
  CL993_p95[i,] = quantile(CL99_R3[i,], c(.05, .95))
}


# 05 - Graphics spectra ----

  # Display detrend procedure
par(mfrow=c(3,1))
plot (dat,type="l",col="black",xlab="Level (m)",ylab="MS (m3/kg)")
lines(dat[,1],trend,col="red")
plot(dat[,1],sigD,type="l",col="black",xlab="Level (m)",ylab="MS ave-det")
lines(InstAmp,col="red")
lines(TrAmp,col="forestgreen")
plot(dat[,1],sigDN,typ="l",col="black",xlab="Level (m)",ylab="MS var-det")

  # Display data + one simulation at 15% uncertainty
par(mfrow=c(1,1))
plot(dat, type="l", xlab="Level (m)", ylab="MS (m3/kg)")
lines(tr3[,1],dat[,2],col="red")
title(main="Comparison between the original series and one of the randomized series")

  # Display spectrum of the original series
par(mfrow=c(1,1))
plot (freq,PowC, type="l",col="black",xlab="Frequency (cycles/m)",
      ylab="Power",xlim=c(0,1/(2*dtmoy)),ylim=c(0,1.2*max(PowC)))
lines(freq,CL50,col="forestgreen")
lines(freq,CL90,col="darkorange")
lines(freq,CL95,col="red")
lines(freq,CL99,col="darkblue")
title(main="Lomb-Scargle raw and red-noise corrected spectra")

par(mfrow=c(2,3))
plot (FreqMean1,powC1_p95[,1],col="lightgrey",type="l", 
      xlab="Frequency (cycles/m)",ylab="Power",
      xlim=c(0,max(FreqMean1)),ylim=c(0,1.2*max(PowC)))
lines(FreqMean1,powC1_p95[,2],col="lightgrey")
lines(FreqMean1,PcorMean1    ,col="black")
lines(FreqMean1,CL501_p95[,1],col="forestgreen",lty=2)
lines(FreqMean1,CL501_p95[,2],col="forestgreen",lty=2)
lines(FreqMean1,CL901_p95[,1],col="darkorange",lty=2)
lines(FreqMean1,CL901_p95[,2],col="darkorange",lty=2)
lines(FreqMean1,CL951_p95[,1],col="red",lty=2)
lines(FreqMean1,CL951_p95[,2],col="red",lty=2)
lines(FreqMean1,CL991_p95[,1],col="darkblue",lty=2)
lines(FreqMean1,CL991_p95[,2],col="darkblue",lty=2)
lines(FreqMean1,CL50Mean1    ,col="forestgreen")
lines(FreqMean1,CL90Mean1    ,col="darkorange")
lines(FreqMean1,CL95Mean1    ,col="red")
lines(FreqMean1,CL99Mean1    ,col="darkblue")
title("Lomb-Scargle spectrum - 5% uncertainty on the sample position")

plot (FreqMean2,powC2_p95[,1],col="lightgrey",type="l",
      xlab="Frequency (cycles/m)", ylab="Power",
      xlim=c(0,max(FreqMean2)),ylim=c(0,1.2*max(PowC)))
lines(FreqMean2,powC2_p95[,2],col="lightgrey")
lines(FreqMean2,PcorMean2    ,col="black")
lines(FreqMean2,CL502_p95[,1],col="forestgreen",lty=2)
lines(FreqMean2,CL502_p95[,2],col="forestgreen",lty=2)
lines(FreqMean2,CL902_p95[,1],col="darkorange",lty=2)
lines(FreqMean2,CL902_p95[,2],col="darkorange",lty=2)
lines(FreqMean2,CL952_p95[,1],col="red",lty=2)
lines(FreqMean2,CL952_p95[,2],col="red",lty=2)
lines(FreqMean2,CL992_p95[,1],col="darkblue",lty=2)
lines(FreqMean2,CL992_p95[,2],col="darkblue",lty=2)
lines(FreqMean2,CL50Mean2    ,col="forestgreen")
lines(FreqMean2,CL90Mean2    ,col="darkorange")
lines(FreqMean2,CL95Mean2    ,col="red")
lines(FreqMean2,CL99Mean2    ,col="darkblue")
title("Lomb-Scargle spectrum - 10% uncertainty on the sample position")

plot (FreqMean3,powC3_p95[,1],col="lightgrey",type="l",
      xlab="Frequency (cycles/m)", ylab="Power",
      xlim=c(0,max(FreqMean3)),ylim=c(0,1.2*max(PowC)))
lines(FreqMean3,powC3_p95[,2],col="lightgrey")
lines(FreqMean3,PcorMean3    ,col="black")
lines(FreqMean3,CL503_p95[,1],col="forestgreen",lty=2)
lines(FreqMean3,CL503_p95[,2],col="forestgreen",lty=2)
lines(FreqMean3,CL903_p95[,1],col="darkorange",lty=2)
lines(FreqMean3,CL903_p95[,2],col="darkorange",lty=2)
lines(FreqMean3,CL953_p95[,1],col="red",lty=2)
lines(FreqMean3,CL953_p95[,2],col="red",lty=2)
lines(FreqMean3,CL993_p95[,1],col="darkblue",lty=2)
lines(FreqMean3,CL993_p95[,2],col="darkblue",lty=2)
lines(FreqMean3,CL50Mean3    ,col="forestgreen")
lines(FreqMean3,CL90Mean3    ,col="darkorange")
lines(FreqMean3,CL95Mean3    ,col="red")
lines(FreqMean3,CL99Mean3    ,col="darkblue")
title("Lomb-Scargle spectrum - 15% uncertainty on the sample position")


plot(freq,PowC,type='l',col='black', xlab="Frequency (cycles/m)", ylab="Power")
lines(FreqMean1,PcorMean1,col="red")
title("2pi-MTM spectrum - comparison between original and 5%-time-randomized series")

plot(freq,PowC,type='l',col='black', xlab="Frequency (cycles/m)", ylab="Power")
lines(FreqMean2,PcorMean2,col="red")
title("2pi-MTM spectrum - comparison between original and 10%-time-randomized series")

plot(freq,PowC,type='l',col='black', xlab="Frequency (cycles/m)", ylab="Power")
lines(FreqMean3,PcorMean3,col="red")
title("2pi-MTM spectrum - comparison between original and 15%-time-randomized series")

