# Spectral analysis of the GR series
library(multitaper)
library(dplR)
library(astrochron)

# Working directory under Mac OS X - to adapt
setwd ("~/Desktop/Working_Directory_Path")

# Working directory under Windows - to adapt
setwd ("D://Working_Directory_Path")

DetrendLaCharce <- function(dat) {
  ModL    <- lm(dat[,2]~dat[,1])
  lin     <- predict(ModL,data.frame(x = dat[,1]))
  sigD    <- dat[,2]-lin
  sigDN   <- (sigD-mean(sigD))/sd(sigD)
  datDN   <- data.frame(dat[,1],sigDN)
  trend   <- data.frame(dat[,1],lin)
  DLC     <- list(datDN=datDN,trend=trend)
  return(DLC)
}


# 01 - Preparing data ----
dat   <- read.table("LaCharce_GR_R.txt", header=T)
dtmoy <- 0.2
DetLC <- DetrendLaCharce(dat)
datDN <- DetLC$datDN
trend <- DetLC$trend
sigDN <- datDN[,2]

par(mfrow=c(2,1))
plot (dat,type="l",col="black",xlab="Level (m)", ylab="GR (ppm eU)")
lines(trend,col="red")
plot(datDN,type="l",col="black", xlab="Level (m)", ylab="GR detrended")

SpecLCH <- redfit(datDN[,2],datDN[,1], tType="time", nsim=1000, mctest=TRUE, 
                  ofac=1,hifac=1, n50=1, p = c(0.10, 0.05, 0.01), iwin=0, 
                  txOrdered = TRUE)
freq    <- SpecLCH$freq
PowC    <- SpecLCH$gxxc
CL50    <- SpecLCH$gredth
CL90    <- SpecLCH$gredth * qchisq(0.90,df=2)/2
CL95    <- SpecLCH$gredth * qchisq(0.95,df=2)/2
CL99    <- SpecLCH$gredth * qchisq(0.99,df=2)/2


# 02 - Randomization gamma ----
tim    = dat[,1]
deltaT = tim[2:length(tim)]-tim[1:length(tim)-1]
nsimul = 10 # Number of simulations: to adapt
d1     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d2     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d3     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
for (i in 1:length(deltaT)) {
  ave  = deltaT[i]
  s1   = 0.05*ave
  s2   = 0.10*ave
  s3   = 0.15*ave
  
  v1   = s1^2
  v2   = s2^2
  v3   = s3^2
  
  sc1  = v1/ave
  sc2  = v2/ave
  sc3  = v3/ave
  
  sh1  = ave/sc1
  sh2  = ave/sc2
  sh3  = ave/sc3
  
  d1[i,1:nsimul] = rgamma(n=nsimul,shape=sh1,scale=sc1)
  d2[i,1:nsimul] = rgamma(n=nsimul,shape=sh2,scale=sc2)
  d3[i,1:nsimul] = rgamma(n=nsimul,shape=sh3,scale=sc3)
}

tr1     = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr2     = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr3     = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr1[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr2[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr3[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)

for (j in 1:nsimul) {
  for (i in 1:length(deltaT)) {
    tr1[i+1,j] = tr1[i,j]+d1[i,j]
    tr2[i+1,j] = tr2[i,j]+d2[i,j]
    tr3[i+1,j] = tr3[i,j]+d3[i,j]
  }
}


# 03 - 2pi-MTM spectrum of the time-randomized series ----
loop    = 0
sig     = dat[,2]
nfreq   = length(freq)

freq_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
freq_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
freq_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

PowC_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
PowC_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
PowC_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

CL50_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
CL90_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
CL95_R1 = matrix(rep(0,nfreq),nfreq,nsimul)
CL99_R1 = matrix(rep(0,nfreq),nfreq,nsimul)

CL50_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
CL90_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
CL95_R2 = matrix(rep(0,nfreq),nfreq,nsimul)
CL99_R2 = matrix(rep(0,nfreq),nfreq,nsimul)

CL50_R3 = matrix(rep(0,nfreq),nfreq,nsimul)
CL90_R3 = matrix(rep(0,nfreq),nfreq,nsimul)
CL95_R3 = matrix(rep(0,nfreq),nfreq,nsimul)
CL99_R3 = matrix(rep(0,nfreq),nfreq,nsimul)

lres1   = matrix(rep(0,nsimul),1,nsimul)
lres2   = matrix(rep(0,nsimul),1,nsimul)
lres3   = matrix(rep(0,nsimul),1,nsimul)

for (i in 1:nsimul) {
  s        = Sys.time()
  loop     = loop+1
  print(loop)
  trand1   = tr1[,i]
  trand2   = tr2[,i]
  trand3   = tr3[,i]
  
  datR1    = data.frame(trand1,sig)
  datR2    = data.frame(trand2,sig)
  datR3    = data.frame(trand3,sig)
  
  detLC_R1 = DetrendLaCharce(datR1)
  detLC_R2 = DetrendLaCharce(datR2)
  detLC_R3 = DetrendLaCharce(datR3)
  
  SpecLCH_R1  = redfit(detLC_R1$datDN[,2],detLC_R1$datDN[,1], 
                       tType="time", nsim=1000, mctest=TRUE, 
                       ofac=1,hifac=1, n50=1, p = c(0.10, 0.05, 0.01), 
                       iwin=0, txOrdered = TRUE)
  freq_R1[1:length(SpecLCH_R1$freq),i] = SpecLCH_R1$freq
  PowC_R1[1:length(SpecLCH_R1$freq),i] = SpecLCH_R1$gxxc
  CL50_R1[1:length(SpecLCH_R1$freq),i] = SpecLCH_R1$gredth
  CL90_R1[1:length(SpecLCH_R1$freq),i] = SpecLCH_R1$gredth * qchisq(0.90,df=2)/2
  CL95_R1[1:length(SpecLCH_R1$freq),i] = SpecLCH_R1$gredth * qchisq(0.95,df=2)/2
  CL99_R1[1:length(SpecLCH_R1$freq),i] = SpecLCH_R1$gredth * qchisq(0.99,df=2)/2
  
  SpecLCH_R2  = redfit(detLC_R2$datDN[,2],detLC_R2$datDN[,1],
                       tType="time", nsim=1000, mctest=TRUE, 
                       ofac=1,hifac=1, n50=1, p = c(0.10, 0.05, 0.01), 
                       iwin=0, txOrdered = TRUE)
  freq_R2[1:length(SpecLCH_R2$freq),i] = SpecLCH_R2$freq
  PowC_R2[1:length(SpecLCH_R2$freq),i] = SpecLCH_R2$gxxc
  CL50_R2[1:length(SpecLCH_R2$freq),i] = SpecLCH_R2$gredth
  CL90_R2[1:length(SpecLCH_R2$freq),i] = SpecLCH_R2$gredth * qchisq(0.90,df=2)/2
  CL95_R2[1:length(SpecLCH_R2$freq),i] = SpecLCH_R2$gredth * qchisq(0.95,df=2)/2
  CL99_R2[1:length(SpecLCH_R2$freq),i] = SpecLCH_R2$gredth * qchisq(0.99,df=2)/2
  
  SpecLCH_R3  = redfit(detLC_R3$datDN[,2],detLC_R3$datDN[,1],
                       tType="time", nsim=1000, mctest=TRUE, 
                       ofac=1,hifac=1, n50=1, p = c(0.10, 0.05, 0.01),
                       iwin=0, txOrdered = TRUE)
  freq_R3[1:length(SpecLCH_R3$freq),i] = SpecLCH_R3$freq
  PowC_R3[1:length(SpecLCH_R3$freq),i] = SpecLCH_R3$gxxc
  CL50_R3[1:length(SpecLCH_R3$freq),i] = SpecLCH_R3$gredth
  CL90_R3[1:length(SpecLCH_R3$freq),i] = SpecLCH_R3$gredth * qchisq(0.90,df=2)/2
  CL95_R3[1:length(SpecLCH_R3$freq),i] = SpecLCH_R3$gredth * qchisq(0.95,df=2)/2
  CL99_R3[1:length(SpecLCH_R3$freq),i] = SpecLCH_R3$gredth * qchisq(0.99,df=2)/2
  
  lres1[i]    = length(SpecLCH_R1$freq)
  lres2[i]    = length(SpecLCH_R2$freq)
  lres3[i]    = length(SpecLCH_R3$freq)
  a           = Sys.time()
  TimElaps    = a-s
  print(TimElaps)
}


# 04 - Calculating the average spectrum and relative error ----

FreqMean1  = matrix(rep(0,min(lres1))  ,nrow=min(lres1),ncol=1)
PcorMean1  = matrix(rep(0,min(lres1))  ,nrow=min(lres1),ncol=1)
CL50Mean1  = matrix(rep(0,min(lres1))  ,nrow=min(lres1),ncol=1)
CL90Mean1  = matrix(rep(0,min(lres1))  ,nrow=min(lres1),ncol=1)
CL95Mean1  = matrix(rep(0,min(lres1))  ,nrow=min(lres1),ncol=1)
CL99Mean1  = matrix(rep(0,min(lres1))  ,nrow=min(lres1),ncol=1)
powC1_p95  = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)
CL501_p95  = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)
CL901_p95  = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)
CL951_p95  = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)
CL991_p95  = matrix(rep(0,2*min(lres1)),nrow=min(lres1),ncol=2)

FreqMean2  = matrix(rep(0,min(lres2))  ,nrow=min(lres2),ncol=1)
PcorMean2  = matrix(rep(0,min(lres2))  ,nrow=min(lres2),ncol=1)
CL50Mean2  = matrix(rep(0,min(lres2))  ,nrow=min(lres2),ncol=1)
CL90Mean2  = matrix(rep(0,min(lres2))  ,nrow=min(lres2),ncol=1)
CL95Mean2  = matrix(rep(0,min(lres2))  ,nrow=min(lres2),ncol=1)
CL99Mean2  = matrix(rep(0,min(lres2))  ,nrow=min(lres2),ncol=1)
powC2_p95  = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)
CL502_p95  = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)
CL902_p95  = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)
CL952_p95  = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)
CL992_p95  = matrix(rep(0,2*min(lres2)),nrow=min(lres2),ncol=2)

FreqMean3  = matrix(rep(0,min(lres3))  ,nrow=min(lres3),ncol=1)
PcorMean3  = matrix(rep(0,min(lres3))  ,nrow=min(lres3),ncol=1)
CL50Mean3  = matrix(rep(0,min(lres3))  ,nrow=min(lres3),ncol=1)
CL90Mean3  = matrix(rep(0,min(lres3))  ,nrow=min(lres3),ncol=1)
CL95Mean3  = matrix(rep(0,min(lres3))  ,nrow=min(lres3),ncol=1)
CL99Mean3  = matrix(rep(0,min(lres3))  ,nrow=min(lres3),ncol=1)
powC3_p95  = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)
CL503_p95  = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)
CL903_p95  = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)
CL953_p95  = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)
CL993_p95  = matrix(rep(0,2*min(lres3)),nrow=min(lres3),ncol=2)

for (i in 1:min(lres1)) {
  FreqMean1[i]  = mean(freq_R1[i,])
  PcorMean1[i]  = mean(PowC_R1[i,])
  CL50Mean1[i]  = mean(CL50_R1[i,])
  CL90Mean1[i]  = mean(CL90_R1[i,])
  CL95Mean1[i]  = mean(CL95_R1[i,])
  CL99Mean1[i]  = mean(CL99_R1[i,])
  powC1_p95[i,] = quantile(PowC_R1[i,], c(.05, .95))
  CL501_p95[i,] = quantile(CL50_R1[i,], c(.05, .95))
  CL901_p95[i,] = quantile(CL90_R1[i,], c(.05, .95))
  CL951_p95[i,] = quantile(CL95_R1[i,], c(.05, .95))
  CL991_p95[i,] = quantile(CL99_R1[i,], c(.05, .95))
}

for (i in 1:min(lres2)) {
  FreqMean2[i]  = mean(freq_R2[i,])
  PcorMean2[i]  = mean(PowC_R2[i,])
  CL50Mean2[i]  = mean(CL50_R2[i,])
  CL90Mean2[i]  = mean(CL90_R2[i,])
  CL95Mean2[i]  = mean(CL95_R2[i,])
  CL99Mean2[i]  = mean(CL99_R2[i,])
  powC2_p95[i,] = quantile(PowC_R2[i,], c(.05, .95))
  CL502_p95[i,] = quantile(CL50_R2[i,], c(.05, .95))
  CL902_p95[i,] = quantile(CL90_R2[i,], c(.05, .95))
  CL952_p95[i,] = quantile(CL95_R2[i,], c(.05, .95))
  CL992_p95[i,] = quantile(CL99_R2[i,], c(.05, .95))
}

for (i in 1:min(lres3)) {
  FreqMean3[i]  = mean(freq_R3[i,])
  PcorMean3[i]  = mean(PowC_R3[i,])
  CL50Mean3[i]  = mean(CL50_R3[i,])
  CL90Mean3[i]  = mean(CL90_R3[i,])
  CL95Mean3[i]  = mean(CL95_R3[i,])
  CL99Mean3[i]  = mean(CL99_R3[i,])
  powC3_p95[i,] = quantile(PowC_R3[i,], c(.05, .95))
  CL503_p95[i,] = quantile(CL50_R3[i,], c(.05, .95))
  CL903_p95[i,] = quantile(CL90_R3[i,], c(.05, .95))
  CL953_p95[i,] = quantile(CL95_R3[i,], c(.05, .95))
  CL993_p95[i,] = quantile(CL99_R3[i,], c(.05, .95))
}


# 05 - Graphics spectra ----

par(mfrow=c(1,1))
plot (freq,PowC, type="l", col="black", xlab="Frequency (cycles/m)", 
      ylab="Power",xlim=c(0,2.5),ylim=c(0,1.2*max(PowC)))
lines(freq,CL50,col="forestgreen")
lines(freq,CL90,col="darkorange")
lines(freq,CL95,col="red")
lines(freq,CL99,col="darkblue")
title(main="Lomb-Scargle red-noise corrected spectrum")

par(mfrow=c(2,3))
plot (FreqMean1,powC1_p95[,1],col="lightgrey",type="l", 
      xlab="Frequency (cycles/m)",ylab="Power",
      xlim=c(0,max(FreqMean1)),ylim=c(0,1.2*max(PowC)))
lines(FreqMean1,powC1_p95[,2],col="lightgrey")
lines(FreqMean1,PcorMean1    ,col="black")
lines(FreqMean1,CL501_p95[,1],col="forestgreen",lty=2)
lines(FreqMean1,CL501_p95[,2],col="forestgreen",lty=2)
lines(FreqMean1,CL901_p95[,1],col="darkorange",lty=2)
lines(FreqMean1,CL901_p95[,2],col="darkorange",lty=2)
lines(FreqMean1,CL951_p95[,1],col="red",lty=2)
lines(FreqMean1,CL951_p95[,2],col="red",lty=2)
lines(FreqMean1,CL991_p95[,1],col="darkblue",lty=2)
lines(FreqMean1,CL991_p95[,2],col="darkblue",lty=2)
lines(FreqMean1,CL50Mean1    ,col="forestgreen")
lines(FreqMean1,CL90Mean1    ,col="darkorange")
lines(FreqMean1,CL95Mean1    ,col="red")
lines(FreqMean1,CL99Mean1    ,col="darkblue")
title("Lomb-Scargle spectrum - 5% uncertainty on the sample position")

plot (FreqMean2,powC2_p95[,1],col="lightgrey",type="l",
      xlab="Frequency (cycles/m)", ylab="Power",
      xlim=c(0,max(FreqMean2)),ylim=c(0,1.2*max(PowC)))
lines(FreqMean2,powC2_p95[,2],col="lightgrey")
lines(FreqMean2,PcorMean2    ,col="black")
lines(FreqMean2,CL502_p95[,1],col="forestgreen",lty=2)
lines(FreqMean2,CL502_p95[,2],col="forestgreen",lty=2)
lines(FreqMean2,CL902_p95[,1],col="darkorange",lty=2)
lines(FreqMean2,CL902_p95[,2],col="darkorange",lty=2)
lines(FreqMean2,CL952_p95[,1],col="red",lty=2)
lines(FreqMean2,CL952_p95[,2],col="red",lty=2)
lines(FreqMean2,CL992_p95[,1],col="darkblue",lty=2)
lines(FreqMean2,CL992_p95[,2],col="darkblue",lty=2)
lines(FreqMean2,CL50Mean2    ,col="forestgreen")
lines(FreqMean2,CL90Mean2    ,col="darkorange")
lines(FreqMean2,CL95Mean2    ,col="red")
lines(FreqMean2,CL99Mean2    ,col="darkblue")
title("Lomb-Scargle spectrum - 10% uncertainty on the sample position")

plot (FreqMean3,powC3_p95[,1],col="lightgrey",type="l",
      xlab="Frequency (cycles/m)", ylab="Power",
      xlim=c(0,max(FreqMean3)),ylim=c(0,1.2*max(PowC)))
lines(FreqMean3,powC3_p95[,2],col="lightgrey")
lines(FreqMean3,PcorMean3    ,col="black")
lines(FreqMean3,CL503_p95[,1],col="forestgreen",lty=2)
lines(FreqMean3,CL503_p95[,2],col="forestgreen",lty=2)
lines(FreqMean3,CL903_p95[,1],col="darkorange",lty=2)
lines(FreqMean3,CL903_p95[,2],col="darkorange",lty=2)
lines(FreqMean3,CL953_p95[,1],col="red",lty=2)
lines(FreqMean3,CL953_p95[,2],col="red",lty=2)
lines(FreqMean3,CL993_p95[,1],col="darkblue",lty=2)
lines(FreqMean3,CL993_p95[,2],col="darkblue",lty=2)
lines(FreqMean3,CL50Mean3    ,col="forestgreen")
lines(FreqMean3,CL90Mean3    ,col="darkorange")
lines(FreqMean3,CL95Mean3    ,col="red")
lines(FreqMean3,CL99Mean3    ,col="darkblue")
title("Lomb-Scargle spectrum - 15% uncertainty on the sample position")


plot(freq,PowC,type='l',col='black',xlab="Frequency (cycles/m)",
     ylab="Power",xlim=c(0,2.5),ylim=c(0,1.2*max(PowC)))
lines(FreqMean1,PcorMean1,col="red")
title("2pi-MTM spectrum - comparison between original and 5%-time-randomized series")

plot(freq,PowC,type='l',col='black',xlab="Frequency (cycles/m)",
     ylab="Power",xlim=c(0,2.5),ylim=c(0,1.2*max(PowC)))
lines(FreqMean2,PcorMean2,col="red")
title("2pi-MTM spectrum - comparison between original and 10%-time-randomized series")

plot(freq,PowC,type='l',col='black',xlab="Frequency (cycles/m)",
     ylab="Power",xlim=c(0,2.5),ylim=c(0,1.2*max(PowC)))
lines(FreqMean3,PcorMean3,col="red")
title("2pi-MTM spectrum - comparison between original and 15%-time-randomized series")
