# Spectral analysis of the GR series
library(astrochron)
library(multitaper)

# Working directories Mac OSX
setwd ("~/Desktop/Working_Directory_Path")
source("~/Desktop/Working_Directory_Path/opt_linterp.R")

# Working directories PC
setwd ("D://Working_Directory_Path")
source("D://Working_Directory_Path/opt_linterp.R")

# 01 - Preparing data ----
frq   <- seq(from=0.02, to=0.48, by=0.02)
tim   <- seq(from=1, to=600, by=1)
dt    <- 1
sig   <- 0
for (i in 1:length(frq)) {
  print(frq[i])
  sig <- sig+sin(2*pi*frq[i]*tim)
}

n     <- length(sig)
npad  <- 2^ceiling(log2(n))

spad  <- matrix(rep(0,npad,1),nrow=npad,ncol=1)
spad[1:n] <- sig


# 02 - 2pi-MTM with confidence levels ML96 Tukey's endpoint rule ----

dof   <- 2*3
sigTS <- as.ts(spad)
Spec  <- spec.mtm(sigTS, nw=2, k=3, Ftest=F,nFFT=npad, taper=c("dpss"), 
                  centre=c("none"), jackknife=F, returnZeroFreq=F, plot=F)
freq  <- Spec$freq
pow   <- Spec$spec/npad


# 03 - Randomization gamma ----
deltaT  <- tim[2:length(tim)]-tim[1:length(tim)-1]
nsimul  <- 10
d1      <- matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d2      <- matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d3      <- matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
for (i in 1:length(deltaT)) {
  ave   <- deltaT[i]
  s1    <- 0.05*ave
  s2    <- 0.10*ave
  s3    <- 0.15*ave
  
  v1    <- s1^2
  v2    <- s2^2
  v3    <- s3^2
  
  sc1   <- v1/ave
  sc2   <- v2/ave
  sc3   <- v3/ave
  
  sh1   <- ave/sc1
  sh2   <- ave/sc2
  sh3   <- ave/sc3

  d1[i,1:nsimul] <- rgamma(n=nsimul,shape=sh1,scale=sc1)
  d2[i,1:nsimul] <- rgamma(n=nsimul,shape=sh2,scale=sc2)
  d3[i,1:nsimul] <- rgamma(n=nsimul,shape=sh3,scale=sc3)
}
  
tr1     <- matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr2     <- matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr3     <- matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr1[1,] <- matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr2[1,] <- matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr3[1,] <- matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)

for (j in 1:nsimul) {
  for (i in 1:length(deltaT)) {
    tr1[i+1,j] <- tr1[i,j]+d1[i,j]
    tr2[i+1,j] <- tr2[i,j]+d2[i,j]
    tr3[i+1,j] <- tr3[i,j]+d3[i,j]
  }
}


# 04 - 2pi-MTM spectrum of the time-randomized series ----
loop   <- 0
tri    <- seq(from=min(tim),to=min(tim)+(npad-1)*dt,by=dt)
powR1  <- matrix(rep(0,length(freq)*nsimul),length(freq),nsimul)
powR2  <- matrix(rep(0,length(freq)*nsimul),length(freq),nsimul)
powR3  <- matrix(rep(0,length(freq)*nsimul),length(freq),nsimul)
  for (i in 1:nsimul) {
    loop      <- loop+1
    print(loop)
    trand1    <- tr1[,i]
    trand2    <- tr2[,i]
    trand3    <- tr3[,i]
    
    datR1     <- data.frame(trand1,sig)
    datR2     <- data.frame(trand2,sig)
    datR3     <- data.frame(trand3,sig)

    datRI1    <- opt_linterp(datR1,dt=1,2)
    datRI2    <- opt_linterp(datR2,dt=1,2)
    datRI3    <- opt_linterp(datR3,dt=1,2)
    
    sigRI1    <- datRI1[,2]
    sigRI2    <- datRI2[,2]
    sigRI3    <- datRI3[,2]
    
    sigR1pad  <- matrix(rep(0,npad),npad,1)
    sigR2pad  <- matrix(rep(0,npad),npad,1)
    sigR3pad  <- matrix(rep(0,npad),npad,1)
    
    sigR1pad[1:length(sigRI1),1] <- sigRI1
    sigR2pad[1:length(sigRI2),1] <- sigRI2
    sigR3pad[1:length(sigRI3),1] <- sigRI3
    
    sigR1TS   <- as.ts(sigR1pad)
    sigR2TS   <- as.ts(sigR2pad)
    sigR3TS   <- as.ts(sigR3pad)  
    
    SpecR1    <- spec.mtm(sigR1TS, nw=2, k=3, Ftest=F,nFFT=npad, taper=c("dpss"), 
                            centre=c("none"), jackknife=F, returnZeroFreq=F, plot=F)
    SpecR2    <- spec.mtm(sigR2TS, nw=2, k=3, Ftest=F,nFFT=npad, taper=c("dpss"), 
                            centre=c("none"), jackknife=F, returnZeroFreq=F, plot=F)
    SpecR3    <- spec.mtm(sigR3TS, nw=2, k=3, Ftest=F,nFFT=npad, taper=c("dpss"), 
                            centre=c("none"), jackknife=F, returnZeroFreq=F, plot=F)
    
    powR1[,i] <- SpecR1$spec/npad
    powR2[,i] <- SpecR2$spec/npad
    powR3[,i] <- SpecR3$spec/npad
  }


# 05 - Calculating the average spectrum and relative error ----
SpecMean1      <- matrix(rep(0,length(freq)),nrow=length(freq),ncol=1)
SpecMean2      <- matrix(rep(0,length(freq)),nrow=length(freq),ncol=1)
SpecMean3      <- matrix(rep(0,length(freq)),nrow=length(freq),ncol=1)

powR1_p95      = matrix(rep(0,length(freq)),nrow=length(freq),ncol=2)
powR2_p95      = matrix(rep(0,length(freq)),nrow=length(freq),ncol=2)
powR3_p95      = matrix(rep(0,length(freq)),nrow=length(freq),ncol=2)

for (i in 1:length(freq)) {
  SpecMean1[i] <- mean(powR1[i,])
  SpecMean2[i] <- mean(powR2[i,])
  SpecMean3[i] <- mean(powR3[i,])
  
  powR1_p95[i,] = quantile(powR1[i,], c(.05, .95))
  powR2_p95[i,] = quantile(powR2[i,], c(.05, .95))
  powR3_p95[i,] = quantile(powR3[i,], c(.05, .95))
}

indfmax        <- matrix(rep(0,length(frq)),nrow=length(frq),ncol=1)
for (i in 1:length(frq)) {
  diff         <- abs(freq-frq[i])
  mdiff        <- min(diff)
  indfmax[i]   <- match(mdiff,diff)
}

ErrRel1   <- 100*abs(SpecMean1[indfmax]-pow[indfmax])/pow[indfmax]
ErrRel2   <- 100*abs(SpecMean2[indfmax]-pow[indfmax])/pow[indfmax]
ErrRel3   <- 100*abs(SpecMean3[indfmax]-pow[indfmax])/pow[indfmax]

Discrepancy1 = (powR1_p95[indfmax,2]-powR1_p95[indfmax,1])/(2*SpecMean1[indfmax])
Discrepancy2 = (powR2_p95[indfmax,2]-powR2_p95[indfmax,1])/(2*SpecMean2[indfmax])
Discrepancy3 = (powR3_p95[indfmax,2]-powR3_p95[indfmax,1])/(2*SpecMean3[indfmax])

IndFrst1  <- min(which(ErrRel1>10))
IndFrst2  <- min(which(ErrRel2>10))
IndFrst3  <- min(which(ErrRel3>10))

FirstFr1  <- freq[indfmax[IndFrst1]]
FirstFr2  <- freq[indfmax[IndFrst2]]
FirstFr3  <- freq[indfmax[IndFrst3]]


# 06 - Graphics spectra ----

  # 2pi-MTM spectrum - original + randomized series
par  (mfrow=c(1,4))
plot (freq,pow, type='l', col='black', xlab="Frequency (cycles/m)", 
      ylab="Power", xlim=c(0,0.5))
#title(main="2pi-MTM spectrum with ML96 CL modified Tukey's end point rule")

plot (freq,SpecMean1,type="l",col="black", xlab="Frequency (cycles/m)", ylab="Power")
lines(freq,powR1_p95[,1],col="lightgrey",lty=2)
lines(freq,powR1_p95[,2],col="lightgrey",lty=2)
lines(freq,SpecMean1,col="red")
#title("2pi-MTM spectrum - 5% uncertainty on the sample position")

plot(freq,SpecMean2,type="l",col="black", xlab="Frequency (cycles/m)", ylab="Power")
lines(freq,powR2_p95[,1],col="lightgrey",lty=2)
lines(freq,powR2_p95[,2],col="lightgrey",lty=2)
lines(freq,SpecMean2,col="darkorange")
#title("2pi-MTM spectrum - 10% uncertainty on the sample position")

plot(freq,SpecMean3,type="l",col="black", xlab="Frequency (cycles/m)", ylab="Power")
lines(freq,powR3_p95[,1],col="lightgrey",lty=2)
lines(freq,powR3_p95[,2],col="lightgrey",lty=2)
lines(freq,SpecMean3,col="darkorange4")
#title("2pi-MTM spectrum - 15% uncertainty on the sample position")
