# Spectral analysis of the MS series
library(astrochron)
library(multitaper)

# Set working directory under Mac OSX - to adapt
setwd ("~/Desktop/Working_Directory_Path")
source("~/Desktop/Working_Directory_Path/opt_linterp.R")

# Set working directory under Windows - to adapt
setwd ("D://Working_Directory_Path")
source("D://Working_Directory_Path/opt_linterp.R")

DetrendLaThure <- function(dat,dtmoy) {
  time  = dat[,1]
  sig   = dat[,2]
  ndat  = length(dat[,1])
  dat1  = dat[1:333,]
  dat2  = dat[334:371,]
  dat3  = dat[372:ndat,]
  
  ModL1 = lm(dat1[,2]~dat1[,1])
  ModL2 = lm(dat2[,2]~dat2[,1])
  ModL3 = lm(dat3[,2]~dat3[,1])
  
  lin1  = predict(ModL1,data.frame(x = dat1[,1]))
  lin2  = predict(ModL2,data.frame(x = dat2[,1]))
  lin3  = predict(ModL3,data.frame(x = dat3[,1]))
  
  sigD1 = dat1[,2]-lin1
  sigD2 = dat2[,2]-lin2
  sigD3 = dat3[,2]-lin3
  
  sigD  = matrix(rep(0,ndat),nrow=ndat,ncol=1)
  lin   = matrix(rep(0,ndat),nrow=ndat,ncol=1)
  sigD[1:333]    = sigD1
  sigD[334:371]  = sigD2
  sigD[372:ndat] = sigD3
  lin [1:333]    = lin1
  lin [334:371]  = lin2
  lin [372:ndat] = lin3
  datD           = data.frame(time,sigD)
  AveTrend       = data.frame(time,lin)
  datID          = opt_linterp(datD,dtmoy,2)
  iamp           = hilbert(datID,padfac=1,demean=T,detrend=F,output=T,addmean=T,genplot=F,verbose=F)
  RegAmp         = lowess(iamp,f=0.1)
  sigAD          = datID[,2]/RegAmp$y
  sigADN         = (sigAD-mean(sigAD))/sd(sigAD)
  datADN         = data.frame(datID[,1],sigADN)
  detLT          = list(datDet=datADN,datAveDet=datID,AveTrend=AveTrend,iamp=iamp,VarTrend=RegAmp)
  return(detLT)
  
}

dat    = read.table("LaThure_MS_R.txt", header=T)

# 01 - Preparing data ----
dtmoy  = 0.38
FNy    = 1/(2*dtmoy)
detLT = DetrendLaThure(dat,dtmoy)

ti     = detLT$datDet[,1]
sigIDN = detLT$datDet[,2]

n       = length(sigIDN)
npad    = 2^ceiling(log2(n))

sig_pad = matrix(rep(0,npad,1),nrow=npad,ncol=1)
sig_pad[1:n] = sigIDN
tpad    = seq(from=min(ti),to=(min(ti)+(npad-1)*dtmoy),by=dtmoy)
dat_pad = data.frame(tpad,sig_pad)


# 02 - 2pi-MTM with confidence levels ML96 Tukey's endpoint rule ----
dof     = 2*3
ML96    = mtmML96(dat_pad,tbw=2,ntap=3,padfac=1,demean=T,detrend=T,medsmooth=0.2,
                  opt=3,linLog=2,siglevel=0.95,output=1,CLpwr=T,xmin=0,xmax=1/(2*dtmoy),
                  sigID=F,pl=2,genplot=F,verbose=F)
nfreq   = length(ML96$Frequency) 

ML96_CL90  = ML96$AR1_fit * qchisq(0.90, df=dof)/dof
ML96_CL95  = ML96$AR1_fit * qchisq(0.95, df=dof)/dof
ML96_CL99  = ML96$AR1_fit * qchisq(0.99, df=dof)/dof

# 03 - Randomization gamma ----
tim    = dat[,1]
deltaT = tim[2:length(tim)]-tim[1:(length(tim)-1)]
nsimul = 10 # Number of simulations: to adapt
d1     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d2     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
d3     = matrix(rep(0,length(deltaT)*nsimul),nrow=length(deltaT),ncol=nsimul)
for (i in 1:length(deltaT)) {
  ave  = deltaT[i]
  s1   = 0.05*ave
  s2   = 0.10*ave
  s3   = 0.15*ave
  
  v1   = s1^2
  v2   = s2^2
  v3   = s3^2
  
  sc1  = v1/ave
  sc2  = v2/ave
  sc3  = v3/ave
  
  sh1  = ave/sc1
  sh2  = ave/sc2
  sh3  = ave/sc3
  
  d1[i,1:nsimul] = rgamma(n=nsimul,shape=sh1,scale=sc1)
  d2[i,1:nsimul] = rgamma(n=nsimul,shape=sh2,scale=sc2)
  d3[i,1:nsimul] = rgamma(n=nsimul,shape=sh3,scale=sc3)
}

tr1     = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr2     = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr3     = matrix(rep(0,length(tim),nsimul),nrow=length(tim),ncol=nsimul)
tr1[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr2[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)
tr3[1,] = matrix(rep(tim[1],1,nsimul),nrow=1,ncol=nsimul)

for (j in 1:nsimul) {
  for (i in 1:length(deltaT)) {
    tr1[i+1,j] = tr1[i,j]+d1[i,j]
    tr2[i+1,j] = tr2[i,j]+d2[i,j]
    tr3[i+1,j] = tr3[i,j]+d3[i,j]
  }
}


# 04 - 2pi-MTM spectrum of the time-randomized series ----
loop    = 0
sig     = dat[,2]
powR1   = matrix(rep(0,nfreq*nsimul),nfreq,nsimul)
powR2   = matrix(rep(0,nfreq*nsimul),nfreq,nsimul)
powR3   = matrix(rep(0,nfreq*nsimul),nfreq,nsimul)
CL50_R1 = matrix(rep(0,nfreq*nsimul),nfreq,nsimul)
CL50_R2 = matrix(rep(0,nfreq*nsimul),nfreq,nsimul)
CL50_R3 = matrix(rep(0,nfreq*nsimul),nfreq,nsimul)

  for (i in 1:nsimul) {
    loop         = loop+1
    print(loop)
    trand1       = tr1[,i]
    trand2       = tr2[,i]
    trand3       = tr3[,i]
    
    datR1        = data.frame(trand1,sig)
    datR2        = data.frame(trand2,sig)
    datR3        = data.frame(trand3,sig)
    
    detLT_R1     = DetrendLaThure(datR1,dtmoy)
    detLT_R2     = DetrendLaThure(datR2,dtmoy)
    detLT_R3     = DetrendLaThure(datR3,dtmoy)
    
    datRIDN1     = detLT_R1$datDet
    datRIDN2     = detLT_R2$datDet
    datRIDN3     = detLT_R3$datDet
    
    tri1         = seq(from=min(datRIDN1[,1]),
                       to=min(datRIDN1[,1])+(npad-1)*dtmoy,by=dtmoy)
    tri2         = seq(from=min(datRIDN2[,1]),
                       to=min(datRIDN2[,1])+(npad-1)*dtmoy,by=dtmoy)
    tri3         = seq(from=min(datRIDN3[,1]),
                       to=min(datRIDN3[,1])+(npad-1)*dtmoy,by=dtmoy)
    
    sigRIDN1     = datRIDN1[,2]
    sigRIDN2     = datRIDN2[,2]
    sigRIDN3     = datRIDN3[,2]
    
    sigRIDN1_pad = matrix(rep(0,npad),nrow=npad,ncol=1)
    sigRIDN2_pad = matrix(rep(0,npad),nrow=npad,ncol=1)
    sigRIDN3_pad = matrix(rep(0,npad),nrow=npad,ncol=1)
    
    sigRIDN1_pad[1:length(sigRIDN1),1] = sigRIDN1
    sigRIDN2_pad[1:length(sigRIDN2),1] = sigRIDN2
    sigRIDN3_pad[1:length(sigRIDN3),1] = sigRIDN3
    
    datRIDN1_pad = data.frame(tri1,sigRIDN1_pad)
    datRIDN2_pad = data.frame(tri2,sigRIDN2_pad)
    datRIDN3_pad = data.frame(tri3,sigRIDN3_pad)
    
    ML96_R1      = mtmML96(datRIDN1_pad,tbw=2,ntap=3,padfac=1,demean=T,detrend=T,
                           medsmooth=0.2,opt=3,linLog=2,siglevel=0.95,output=1,
                           CLpwr=T,xmin=0,xmax=1/(2*dtmoy),sigID=F,pl=2,genplot=F,
                           verbose=F)
    ML96_R2      = mtmML96(datRIDN2_pad,tbw=2,ntap=3,padfac=1,demean=T,detrend=T,
                           medsmooth=0.2,opt=3,linLog=2,siglevel=0.95,output=1,
                           CLpwr=T,xmin=0,xmax=1/(2*dtmoy),sigID=F,pl=2,genplot=F,
                           verbose=F)
    ML96_R3      = mtmML96(datRIDN3_pad,tbw=2,ntap=3,padfac=1,demean=T,detrend=T,
                           medsmooth=0.2,opt=3,linLog=2,siglevel=0.95,output=1,
                           CLpwr=T,xmin=0,xmax=1/(2*dtmoy),sigID=F,pl=2,genplot=F,
                           verbose=F)
    CL50_R1[,i]  = ML96_R1$AR1_fit
    CL50_R2[,i]  = ML96_R2$AR1_fit
    CL50_R3[,i]  = ML96_R3$AR1_fit
    
    powR1[,i]    = ML96_R1$Power
    powR2[,i]    = ML96_R2$Power
    powR3[,i]    = ML96_R3$Power
  }

CL90_R1          = CL50_R1 * qchisq(0.90,df=dof)/dof
CL90_R2          = CL50_R2 * qchisq(0.90,df=dof)/dof
CL90_R3          = CL50_R3 * qchisq(0.90,df=dof)/dof

CL95_R1          = CL50_R1 * qchisq(0.95,df=dof)/dof
CL95_R2          = CL50_R2 * qchisq(0.95,df=dof)/dof
CL95_R3          = CL50_R3 * qchisq(0.95,df=dof)/dof

CL99_R1          = CL50_R1 * qchisq(0.99,df=dof)/dof
CL99_R2          = CL50_R2 * qchisq(0.99,df=dof)/dof
CL99_R3          = CL50_R3 * qchisq(0.99,df=dof)/dof


# 05 - Calculating the average spectrum and relative error ----
SpecMean1 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
SpecMean2 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
SpecMean3 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)

CL50Mean1 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
CL50Mean2 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
CL50Mean3 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)

CL90Mean1 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
CL90Mean2 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
CL90Mean3 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)

CL95Mean1 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
CL95Mean2 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
CL95Mean3 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)

CL99Mean1 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
CL99Mean2 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)
CL99Mean3 = matrix(rep(0,nfreq),nrow=nfreq,ncol=1)

powR1_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
powR2_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
powR3_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)

CL501_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
CL502_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
CL503_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)

CL901_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
CL902_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
CL903_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)

CL951_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
CL952_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
CL953_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)

CL991_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
CL992_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)
CL993_p95 = matrix(rep(0,nfreq),nrow=nfreq,ncol=2)

for (i in 1:nfreq) {
  SpecMean1[i] = mean(powR1[i,])
  SpecMean2[i] = mean(powR2[i,])
  SpecMean3[i] = mean(powR3[i,])
  
  CL50Mean1[i] = mean(CL50_R1[i,])
  CL50Mean2[i] = mean(CL50_R2[i,])
  CL50Mean3[i] = mean(CL50_R3[i,])
  
  CL90Mean1[i] = mean(CL90_R1[i,])
  CL90Mean2[i] = mean(CL90_R2[i,])
  CL90Mean3[i] = mean(CL90_R3[i,])
  
  CL95Mean1[i] = mean(CL95_R1[i,])
  CL95Mean2[i] = mean(CL95_R2[i,])
  CL95Mean3[i] = mean(CL95_R3[i,])
  
  CL99Mean1[i] = mean(CL99_R1[i,])
  CL99Mean2[i] = mean(CL99_R2[i,])
  CL99Mean3[i] = mean(CL99_R3[i,])
  
  powR1_p95[i,] = quantile(powR1[i,], c(.05, .95))
  powR2_p95[i,] = quantile(powR2[i,], c(.05, .95))
  powR3_p95[i,] = quantile(powR3[i,], c(.05, .95))
  
  CL501_p95[i,] = quantile(CL50_R1[i,], c(.05, .95))
  CL502_p95[i,] = quantile(CL50_R2[i,], c(.05, .95))
  CL503_p95[i,] = quantile(CL50_R3[i,], c(.05, .95))
  
  CL901_p95[i,] = quantile(CL90_R1[i,], c(.05, .95))
  CL902_p95[i,] = quantile(CL90_R2[i,], c(.05, .95))
  CL903_p95[i,] = quantile(CL90_R3[i,], c(.05, .95))
  
  CL951_p95[i,] = quantile(CL95_R1[i,], c(.05, .95))
  CL952_p95[i,] = quantile(CL95_R2[i,], c(.05, .95))
  CL953_p95[i,] = quantile(CL95_R3[i,], c(.05, .95))
  
  CL991_p95[i,] = quantile(CL99_R1[i,], c(.05, .95))
  CL992_p95[i,] = quantile(CL99_R2[i,], c(.05, .95))
  CL993_p95[i,] = quantile(CL99_R3[i,], c(.05, .95))
}


# 06 - Graphics spectra ----

  # Display detrend procedure
par  (mfrow=c(3,1))
plot (dat,type="l",xlab="Level (m)", ylab="MS (m3/kg)")
lines(detLT$AveTrend,col="red")
plot (detLT$datAveDet,type="l",xlab="Level (m)", ylab="MS ave-det")
lines(detLT$iamp[,1],detLT$iamp[,2],col="red")
lines(detLT$VarTrend$x,detLT$VarTrend$y,col="darkgreen")
plot (detLT$datDet,type="l",xlab="Level (m)", ylab="MS var-det")

  # Display data + one simulation at 15% uncertainty
par(mfrow=c(1,1))
plot(dat, type="l", xlab="Level (m)", ylab="MS (m3/kg)")
lines(tr3[,1],dat[,2],col="red")
title(main="Comparison between the original series and one of the randomized series")

  # Display spectrum of the original series
par(mfrow=c(1,1))
plot (ML96$Frequency,ML96$Power, type='l', col='black', xlab="Frequency (cycles/m)", 
      ylab="Power",lab=c(9,5,0), xlim=c(0,1/(2*dtmoy)))
lines(ML96$Frequency,ML96$AR1_fit,col="forestgreen")
lines(ML96$Frequency,ML96_CL90,col='orange')
lines(ML96$Frequency,ML96_CL95,col='red')
lines(ML96$Frequency,ML96_CL99,col="darkblue")
title(main="2pi-MTM spectrum with ML96 CL modified Tukey's end point rule")

  # Display spectra with 5, 10 and 15% uncertainty
par(mfrow=c(2,3))
plot (ML96$Frequency,powR1_p95[,1],type="l",col="lightgrey",
      xlab="Frequency (cycles/m)",ylab="Power",lab=c(9,5,0),
      xlim=c(0,1/(2*dtmoy)),ylim=c(0,1.2*max(ML96$Power)))
lines(ML96$Frequency,powR1_p95[,2],col="lightgrey")
lines(ML96$Frequency,SpecMean1    ,col="black")
lines(ML96$Frequency,CL501_p95[,1],col="forestgreen",lty=2)
lines(ML96$Frequency,CL501_p95[,2],col="forestgreen",lty=2)
lines(ML96$Frequency,CL901_p95[,1],col="darkorange" ,lty=2)
lines(ML96$Frequency,CL901_p95[,2],col="darkorange" ,lty=2)
lines(ML96$Frequency,CL951_p95[,1],col="red"        ,lty=2)
lines(ML96$Frequency,CL951_p95[,2],col="red"        ,lty=2)
lines(ML96$Frequency,CL991_p95[,1],col="darkblue"   ,lty=2)
lines(ML96$Frequency,CL991_p95[,2],col="darkblue"   ,lty=2)
lines(ML96$Frequency,CL50Mean1    ,col="forestgreen")
lines(ML96$Frequency,CL90Mean1    ,col='darkorange')
lines(ML96$Frequency,CL95Mean1    ,col='red')
lines(ML96$Frequency,CL99Mean1    ,col="darkblue")
title("2pi-MTM spectrum - 5% uncertainty on the sample position")

plot(ML96$Frequency,powR2_p95[,1],type="l",col="lightgrey",
     xlab="Frequency (cycles/m)",ylab="Power",lab=c(9,5,0),
     xlim=c(0,1/(2*dtmoy)),ylim=c(0,1.2*max(ML96$Power)))
lines(ML96$Frequency,powR2_p95[,2],col="lightgrey")
lines(ML96$Frequency,SpecMean2    ,col="black")
lines(ML96$Frequency,CL502_p95[,1],col="forestgreen",lty=2)
lines(ML96$Frequency,CL502_p95[,2],col="forestgreen",lty=2)
lines(ML96$Frequency,CL902_p95[,1],col="darkorange" ,lty=2)
lines(ML96$Frequency,CL902_p95[,2],col="darkorange" ,lty=2)
lines(ML96$Frequency,CL952_p95[,1],col="red"        ,lty=2)
lines(ML96$Frequency,CL952_p95[,2],col="red"        ,lty=2)
lines(ML96$Frequency,CL992_p95[,1],col="darkblue"   ,lty=2)
lines(ML96$Frequency,CL992_p95[,2],col="darkblue"   ,lty=2)
lines(ML96$Frequency,CL50Mean2    ,col="forestgreen")
lines(ML96$Frequency,CL90Mean2    ,col='darkorange')
lines(ML96$Frequency,CL95Mean2    ,col='red')
lines(ML96$Frequency,CL99Mean2    ,col="darkblue")
title("2pi-MTM spectrum - 10% uncertainty on the sample position")

plot(ML96$Frequency,powR3_p95[,1],type="l",col="lightgrey",
     xlab="Frequency (cycles/m)",ylab="Power",lab=c(9,5,0),
     xlim=c(0,1/(2*dtmoy)),ylim=c(0,1.2*max(ML96$Power)))
lines(ML96$Frequency,powR3_p95[,2],col="lightgrey")
lines(ML96$Frequency,SpecMean3,col="black")
lines(ML96$Frequency,CL503_p95[,1],col="forestgreen",lty=2)
lines(ML96$Frequency,CL503_p95[,2],col="forestgreen",lty=2)
lines(ML96$Frequency,CL903_p95[,1],col="darkorange" ,lty=2)
lines(ML96$Frequency,CL903_p95[,2],col="darkorange" ,lty=2)
lines(ML96$Frequency,CL953_p95[,1],col="red"        ,lty=2)
lines(ML96$Frequency,CL953_p95[,2],col="red"        ,lty=2)
lines(ML96$Frequency,CL993_p95[,1],col="darkblue"   ,lty=2)
lines(ML96$Frequency,CL993_p95[,2],col="darkblue"   ,lty=2)
lines(ML96$Frequency,CL50Mean3    ,col="forestgreen")
lines(ML96$Frequency,CL90Mean3    ,col='darkorange')
lines(ML96$Frequency,CL95Mean3    ,col='red')
lines(ML96$Frequency,CL99Mean3    ,col="darkblue")
title("2pi-MTM spectrum - 15% uncertainty on the sample position")

# Comparisons of spectra before and after randomization
plot (ML96$Frequency,ML96$Power,type='l',col='black', xlab="Frequency (cycles/m)", 
      ylab="Power", lab=c(9,5,0), xlim=c(0,1/(2*dtmoy)))
lines(ML96$Frequency,SpecMean1,col="red")
title("2pi-MTM spectrum - comparison between original and 5%-time-randomized series")

plot (ML96$Frequency,ML96$Power,type='l',col='black', xlab="Frequency (cycles/m)", 
      ylab="Power", lab=c(9,5,0), xlim=c(0,1/(2*dtmoy)))
lines(ML96$Frequency,SpecMean2,col="red")
title("2pi-MTM spectrum - comparison between original and 10%-time-randomized series")

plot (ML96$Frequency,ML96$Power,type='l',col='black', xlab="Frequency (cycles/m)", 
      ylab="Power", lab=c(9,5,0), xlim=c(0,1/(2*dtmoy)))
lines(ML96$Frequency,SpecMean3,col="red")
title("2pi-MTM spectrum - comparison between original and 15%-time-randomized series")
