close all
clearvars

RTime=38; %Time of recontruction
res=0.1; %Resolution in degrees
SLchange=100; %Applied sea level change
sfac=5; %Smoothing radius in grid cells
cres=0.5; %Resolution of netcdf output
shiftp=20; %Shift for plotting figures in pixels
ShelfDepth=-100; %Depth of continental shelves without sea level change
ShallowDepth=-100; %Depth of shallow inland seas after sea level change

%Colormap for plotting topography and bathymetry
BaseColors=[0.5 0 1;0 0 0.5;0 0 1;0 1 1;0 0.5 0;1 1 0;1 0 0;0.5 0 0.1;1 0.875 0.75]; NCol=size(BaseColors,1);
Nx=60; x1=linspace(1,Nx,NCol)'; x2=(1:Nx)';
BathyMap=[interp1(x1,BaseColors(:,1),x2) interp1(x1,BaseColors(:,2),x2) interp1(x1,BaseColors(:,3),x2)];
TopoMap=[0.85 0.85 0.85; BathyMap; summer(15); flipud(copper(15)); gray(10); flipud(hot(20))];

%Labels for plotting
lonlab={'180W','150W','120W','90W','60W','30W','0','30E','60E','90E','120E','150E','180E'};
latlab={'80S','60S','40S','20S','0','20N','40N','60N','80N'};
elevticks=-6000:1000:6000;
elevlab={'-6km','-5km','-4km','-3km','-2km','-1km','MSL','1km','2km','3km','4km','5km',''};

%Grid that will be used for the reconstruction
lon=-(180-res/2):res:(180-res/2);
lat=-(90-res/2):res:(90-res/2);
[Lon,Lat]=meshgrid(lon,lat);

%Reconstructed Boundaries Mask
disp('Reading Boundaries Mask')

%Extract Coastline, Continental Shelf and Shallow outlines
Boundaries=ncread(['Geometries/Files/Boundaries' num2str(RTime) '.nc'],'Mask');
LB=Boundaries==3;
SB=Boundaries==2;
SS=Boundaries==1;

MaskMap=[0.5 0.5 0.5;0 0 0;0 0 1;1 1 1];

%Plot the mask image
    scrsz = get(0,'ScreenSize'); figshift=0;
    h=figure('Position',[scrsz(3)/6 scrsz(4)/4 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,Boundaries); set(gca,'ydir','normal')
    colormap(MaskMap);
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'fontsize',12)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r500','Boundaries_gridded') %Save image as png

%Additional Land Ocean Mask
disp('Reading Land Ocean Mask')

Mask_LO=ncread(['Geometries/Files/Masks' num2str(RTime) '.nc'],'Mask_LO');
Mask_Topo=ncread(['Geometries/Files/Masks' num2str(RTime) '.nc'],'Mask_Topo');
Mask_comb=Mask_LO; N_Topo=max(Mask_Topo(:)); %Create one combined field, only for plotting
Mask_comb(Mask_Topo>0)=Mask_Topo(Mask_Topo>0)+1;
Mask_comb(Mask_LO>1)=Mask_LO(Mask_LO>1)+N_Topo;

%Extract additional masks; land (LM), sea (SM), ocean (OM) and shallow sea
LM=Mask_LO==1;
Shallow=Mask_LO==2;
SM=Mask_LO==3;
OM=Mask_LO==4;
%Masks for manipulating Asian topography
EAsiaM=Mask_Topo==1;
LowTibetM=Mask_Topo==2;
TibetM=Mask_Topo==3;
HimalayaM=Mask_Topo==4;

LOMap=[0.5 0.5 0.5;1 1 1;1 1 0;0 0.5 0;1 0.6 0;1 0 0;0 1 1;0 0 1;0 0 0];

%Plot the land ocean mask image
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,Mask_comb); set(gca,'ydir','normal')
    colormap(LOMap);
    hold on; contour(lon,lat,LM,[1 1],'color',[1 1 1])
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'fontsize',12)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r500','LOMask_gridded')

%Mueller Reconstructed Bathymetry
disp('Building Topobathymetry')

%Load transformed mueller bathymetry file
MuellerBathy=ncread(['Mueller' num2str(RTime) 'Ma_cor.nc'],'Band1');
MuellerBathy=MuellerBathy';
MuellerBathy(isnan(MuellerBathy))=1;

%Remove Mediterranean Reconstruction
MuellerBathy(1000:1277,1733:2445)=nan;

%Initialise topobathymetry file and apply Mueller bathymetry
topobathy=nan(size(LB));
topobathy(MuellerBathy<0)=MuellerBathy(MuellerBathy<0); %Exclude regions above sea level

%Ocean Floor Age Reconstruction

age=ncread(['Age_' num2str(RTime) '.00Ma.nc'],'Band1'); age=(age-RTime)';
%Read and apply age grid
OceanFloor=nan(size(age));
%Determine depth using age-depth relationship
OceanFloor(age>0)=-2620-330*sqrt(age(age>0));
OceanFloor(age>90)=-5750;

%Plot ocean bathymetry and shelf boundary contours
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,OceanFloor,[-6100 6000])
    hold on
    contour(lon,lat,LB+SB,[1 1],'Color','k')
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',11)
    title(['Reconstructed Ocean Floor ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step1_' num2str(RTime)])

%Update the bathymetry, keeping mueller only where agegrid is undefined
topobathy(~isnan(OceanFloor))=OceanFloor(~isnan(OceanFloor));

%Plot new bathymetry with missing regions filled by muller reconstruction
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathy,[-6100 6000])
    hold on
    contour(lon,lat,LB+SB,[1 1],'Color','k')
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Reconstructed Ocean Floor using Mueller Data ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step2_' num2str(RTime)])

%Continental Shelves and Shallow Regions

sbathy=ncread(['SS_' num2str(RTime) '.00Ma.nc'],'Band1'); sbathy=sbathy';
sbathy(sbathy<topobathy)=topobathy(sbathy<topobathy); %Remove parts that are deeper than current bathymetry
topobathy(SS==1)=sbathy(SS==1);
topobathy(SB==1)=ShelfDepth;
%Replace continental shelf by shallow region depth where the latter is deeper and less than 2000m
topobathy(((SB==1).*(sbathy>-2000).*(sbathy<ShelfDepth))==1)=sbathy(((SB==1).*(sbathy>-2000).*(sbathy<ShelfDepth))==1);

%Plot current ocean and sea bathymetry
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathy,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Reconstructed Ocean Floor with Continental Shelves and Plateaus ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step3_' num2str(RTime)])

%Reconstructed Topography

topo=ncread(['TopoBR_' num2str(RTime) '.00Ma.nc'],'Band1'); topo=topo';
%Implement adjusted present day topography
topobathy(LB==1)=topo(LB==1);
topobathy((LB.*isnan(topo))==1)=0; %Put elevation to 0 where land mask is defined but not the topography

%Plot the combined topography and shifted present day bathymetry
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathy,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Reconstructed Topobathymetry using Present Day Topography ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step4_' num2str(RTime)])

%Load Adjustment Masks
AdjMask=ncread(['Geometries/Files/AdjMasks' num2str(RTime) '.nc'],'Mask');

% 1: Andes
% 2: Western Sierra
% 3: Greenland
% 4: Middle East
% 5: Kaukasian
% 6: Turkish
% 7: Arabia
% 8: African Rift
% 9: 90E Ridge (N)
% 10: Tasman Rise
% 11: 90E Ridge (S)

%Greenland and Antarctica Isostatic Adjustment

topoI=ncread(['Topo_' num2str(RTime) '.00Ma.nc'],'Band1'); topoI=topoI';
%Correct topography by 30% of difference between current actual and bedrock elevation
Greenland=0.3*(topoI(AdjMask==3)-topo(AdjMask==3));
Greenland(isnan(Greenland))=0; Greenland(Greenland<0)=0;
topobathy(AdjMask==3)=topobathy(AdjMask==3)+Greenland;

Antarctica=0.3*(topoI(1:300,:)-topo(1:300,:)); %Consider everything south of 70deg
Antarctica(isnan(Antarctica))=0; Antarctica(Antarctica<0)=0;
topobathy(1:300,:)=topobathy(1:300,:)+Antarctica;

%Tethys ocean
topobathy(OM==1)=-4500+SLchange;

%Lower Mountain Ranges
Andes=topobathy(AdjMask==1);
%Divide elevation above 1000m by 4
Andes(Andes>1000)=1000+((Andes(Andes>1000))-1000)/4;
topobathy(AdjMask==1)=Andes;

Himalaya=topobathy(HimalayaM); %Use Himalaya mask
Himalaya(Himalaya>2000)=2000+((Himalaya(Himalaya>2000))-2000)/2;
topobathy(HimalayaM)=Himalaya;

Tibet=topobathy(TibetM); %Use Tibet mask
Tibet(Tibet>1500)=1500+((Tibet(Tibet>1500))-1500)/3;
topobathy(TibetM)=Tibet;

LowTibet=topobathy(LowTibetM); %Lowland north of Himalaya
LowTibet=LowTibet/5;
topobathy(LowTibetM)=LowTibet;

EAsia=topobathy(EAsiaM); %Mask surrounding Himalaya and Tibet
EAsia(EAsia>1000)=1000+((EAsia(EAsia>1000))-1000)/4;
topobathy(EAsiaM)=EAsia;

MaskME=AdjMask==4; MaskME(AdjMask==5)=1; MaskME(AdjMask==6)=1; MaskME(AdjMask==7)=1;
ME=topobathy(MaskME); %Middle East topography caused by Africa-Eurasia collision
ME(ME>1000)=1000+((ME(ME>1000))-1000)/2;
topobathy(MaskME)=ME;

Rift=topobathy(AdjMask==8); %West African Rift was not yet formed
Rift(Rift>1000)=1000+((Rift(Rift>1000))-1000)/2;
topobathy(AdjMask==8)=Rift;

WSierra=topobathy(AdjMask==2); %Western Sierras in Northern America were lower
WSierra(WSierra>1000)=1000+((WSierra(WSierra>1000))-1000)/2;
topobathy(AdjMask==2)=WSierra;

%Plot adjusted topobathymetry
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathy,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Reconstructed Topobathymetry with Custom Adjustments ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step5_' num2str(RTime)])

%Save backup of topobathymetry
topobathybu=topobathy;

%Correct Shallow Regions

MaskNER=AdjMask==9; MaskNER(AdjMask==11)=1;
NER=topobathy(MaskNER); %Ninety-East ridge is accentuated
NER(NER>-3000)=-3000+55*sqrt(3000+NER(NER>-3000));
topobathy(MaskNER)=NER;

TR=topobathy(AdjMask==10); %Tasman Rise is made higher
TR(TR>-3000)=-3000+208*(3000+TR(TR>-3000)).^(1/3);
topobathy(AdjMask==10)=TR;
TRenv=topobathy(AdjMask==10); SB_TRenv=SB(AdjMask==10);
TRenv(SB_TRenv==1)=ShelfDepth; %Correct changes to shelves nearby
topobathy(AdjMask==10)=TRenv;
SM_TRenv=SM(AdjMask==10); SM_TRenv(SB_TRenv==1)=0;
SM(AdjMask==10)=SM_TRenv; %Remove extended shelves from Sea Mask

%Wilson Antarctic Reconstruction
WAT=nan(size(topobathy));
topo=ncread(['WilsonAR_' num2str(RTime) '.00Ma.nc'],'Band1');
WAT(1:300,:)=topo'+SLchange;
%Apply (adjusted) Wilson reconstruction but only topography and coasts
topobathy(WAT>ShelfDepth)=WAT(WAT>ShelfDepth);

%Plot topobathymetry including Antarctic reconstruction
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathy,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Reconstructed Topobathymetry using Wilson (2012) Antarctic Topography ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step6_' num2str(RTime)])

%Additional Sea Ocean Mask

%Correct to (final) specified depth regardless of current value, correct for sea level rise
topobathy(((Shallow==1).*(topobathy>ShallowDepth))==1)=ShallowDepth+SLchange;
topobathy(((Shallow==1).*isnan(topobathy))==1)=ShallowDepth+SLchange;
topobathy(SM==1)=-1000+SLchange;
topobathy(OM==1)=-4500+SLchange;

%Plot topobathymetry with mask corrections
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathy,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Reconstructed Topobathymetry ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step7_' num2str(RTime)])

%Find all custom changes made
TBadj=(topobathy-topobathybu)~=0;
TBadj(isnan(topobathybu))=1;
TBadj(isnan(topobathy))=0;
TBadj(((LM==1).*((topobathy-SLchange)<0))==1)=1;

    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathy,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    hold on
    contour(lon,lat,TBadj,[0.5 0.5],'color','k')
    TBadj=double(TBadj); TBadj(TBadj==0)=nan;
    hp=pcolor(lon,lat,TBadj+3000); shading flat; caxis([-6100 6000]);
    alpha(hp,0.25);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Specific Changes ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step7adj_' num2str(RTime)])


%Interpolate Missing Values
disp('Interpolating Missing Values')

edge=10; %Extra edge for interpolation along date line separation
nlon=length(lon); nlat=length(lat); %Redifine nlon and nlat for reconstruction grid
topobathy=[topobathy(:,(nlon-edge+1):nlon) topobathy topobathy(:,1:edge)];
topobathy=inpaintn(topobathy); %Inpaintn algorithm developed by Damien Garcia (Garcia, 2010, Wang et al., 2012)
topobathy=topobathy(:,(edge+1):(edge+nlon)); %Cut off extra edges

%Adjust Global Sea Level
topobathy=topobathy-SLchange;

%Correct Land Mask and fill in Last Gaps

topobathy((LM.*(topobathy<0))==1)=0; %Similar to sea/ocean mask, submerged land put back to 0
topobathy((LM.*isnan(topobathy))==1)=0; %Possible undefined points are put to 0
topobathy(isnan(topobathy))=-4000; %Undifined points in the ocean put to -4000m
topobathy(topobathy<-6000)=-6000; %Points below 6000m depth are corrected

%Plot complete topobathymetry after interpolation and corrections
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathy,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Reconstructed Topobathymetry, Interpolating Missing Values ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step8_' num2str(RTime)])

%Smoothing
disp('Smoothing')
topobathys=zeros(size(topobathy)); %Initialise smoothed topobathymetry

for iz=1:nlon
    for im=1:nlat
        %Define smoothing mask; periodic boundary along date line
        lonvec=mod((iz-sfac):(iz+sfac),nlon); lonvec(lonvec==0)=nlon;
        latvec=max(1,im-sfac):min(nlat,im+sfac);
        topobathys(im,iz)=mean(mean(topobathy(latvec,lonvec)));
    end
end

%Plot smooth topobathymetry
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathys,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    hold on
    contour(lon,lat,topobathys,[0 0],'color','k','linewidth',0.5)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Smoothed Topobathymetry ' num2str(RTime) 'Ma'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step9_' num2str(RTime)])

%Saving
disp('Saving Result')
save(['TopoBathy' num2str(RTime) '.mat'],'topobathys'); %Save as MATLAB binary

%Create Netcdf File to store result at full resolution
ncid = netcdf.create(['TopoBathy' num2str(RTime) '.nc'],'NC_WRITE');
dimidlat = netcdf.defDim(ncid,'latitude',nlat);
dimidlon = netcdf.defDim(ncid,'longitude',nlon);
latitude_ID=netcdf.defVar(ncid,'latitude','double',dimidlat);
longitude_ID=netcdf.defVar(ncid,'longitude','double',dimidlon);
Z_ID = netcdf.defVar(ncid,'Z','double',[dimidlon dimidlat]);
netcdf.endDef(ncid);
netcdf.putVar(ncid,latitude_ID,lat);
netcdf.putVar(ncid,longitude_ID,lon);
netcdf.putVar(ncid,Z_ID,topobathys'); %Store smoothed topobathymetry
netcdf.close(ncid)

%Interpolating to output grid with specified resolution
disp('Interpolating to Desired Grid')
lonc=-(180-cres/2):cres:(180-cres/2); %Defining reduced grid
latc=-(90-cres/2):cres:(90-cres/2);
[Lonc,Latc]=meshgrid(lonc,latc);
topobathyc=interp2(Lon,Lat,topobathys,Lonc,Latc);

%Plot reduced resolution topobathymetry
    h=figure('Position',[scrsz(3)/6+figshift scrsz(4)/4-figshift 2*scrsz(3)/3 2*scrsz(4)/3]); figshift=figshift+shiftp;
    axes('position',[0.05 0.05 0.875 0.9])
    imagesc(lon,lat,topobathyc,[-6100 6000])
    colormap(TopoMap)
    hc=colorbar('position',[0.935 0.05 0.02 0.9]);
    set(hc,'ytick',elevticks)
    set(hc,'yticklabel',elevlab)
    set(gca,'Ydir','normal')
    set(gca,'Xlim',[-180 180])
    set(gca,'Ylim',[-90 90])
    set(gca,'XTick',-180:30:180)
    set(gca,'YTick',-80:20:80)
    set(gca,'XTickLabel',lonlab)
    set(gca,'YTickLabel',latlab)
    grid
    set(gca,'Fontsize',12)
    title(['Smoothed Topobathymetry ' num2str(RTime) 'Ma at ' num2str(cres) 'deg'],'Fontsize',14)
    set(gcf,'paperposition',[0 0 10 6])
    print(h,'-dpng','-r300',['Step10_' num2str(RTime)])


%Create netcdf file at reduced resolution
ncid = netcdf.create(['TopoBathyc' num2str(RTime) '.nc'],'NC_WRITE');
dimidlat = netcdf.defDim(ncid,'latitude',length(latc));
dimidlon = netcdf.defDim(ncid,'longitude',length(lonc));
latitude_ID=netcdf.defVar(ncid,'latitude','double',dimidlat);
longitude_ID=netcdf.defVar(ncid,'longitude','double',dimidlon);
Z_ID = netcdf.defVar(ncid,'Z','double',[dimidlon dimidlat]);
netcdf.endDef(ncid);
netcdf.putVar(ncid,latitude_ID,latc);
netcdf.putVar(ncid,longitude_ID,lonc);
netcdf.putVar(ncid,Z_ID,topobathyc'); %Interpolated topobathymetry field
netcdf.close(ncid)

%End of script