# Read in NetCDF files that contain monthly values.
# The format for the file should be:
#  longitude
#  latitude
#  plus one internally saved variable containing the interesting climate field
# The time variable is not read, I assume the user to know when the instrumental
# records start. Since usually only a few (one or three or so) NetCDFs are used
# I don't need to automate this.
# Author: Joe Werner
###############################################################################

NCDF.Locations <- function( NCDF.DataFile, coordinates){
	# Then, create a grid. Let's assume a regular grid on the lat-lon grid.
	# Set the grid distance and corner coordinates in degrees north and deg east
	# To make things easier: use grid as HadCRUT3
	lonW <- coordinates$lonW
	lonE <- coordinates$lonE;
	latS <- coordinates$latS;
	latN <- coordinates$latN;
	lonIdx <- (NCDF.DataFile$dim[[1]]$vals >= 360+lonW) | (NCDF.DataFile$dim[[1]]$vals <= lonE)
	latIdx <- (NCDF.DataFile$dim[[2]]$vals >= latS) & (NCDF.DataFile$dim[[2]]$vals <= latN)
	
	locs<-matrix(NA, 0, 2);
	for (i in NCDF.DataFile$dim[[1]]$vals[lonIdx]){
		for (j in  NCDF.DataFile$dim[[2]]$vals[latIdx]){
			locs <- rbind(locs, c(j,i));
		}
	}
	return(locs);
}

NCDF.SeasonAvg <- function( TempData, coordinates, StartTime=1850, NA.Value=-99.9){
	library("ncdf")
	#TempData <- "~/Johannes/Giessen/Data/HadCRU/HadCRUT3v.nc";
	
	# Read all the data
	NCDF.DataFile <- open.ncdf(TempData);
	Temperature <- get.var.ncdf(NCDF.DataFile);
	close.ncdf(NCDF.DataFile);
	seasons <- floor(length(Temperature[1,1,])/3);

	lonW <- coordinates$lonW
	lonE <- coordinates$lonE;
	latS <- coordinates$latS;
	latN <- coordinates$latN;
	lonIdx <- cbind(NCDF.DataFile$dim[[1]]$vals >= 360+lonW) | (NCDF.DataFile$dim[[1]]$vals <= lonE)
	latIdx <- (NCDF.DataFile$dim[[2]]$vals >= latS) & (NCDF.DataFile$dim[[2]]$vals <= latN)
	
	locs<-matrix(NA, 0, 2);
	for (i in NCDF.DataFile$dim[[1]]$vals[lonIdx]){
		for (j in  NCDF.DataFile$dim[[2]]$vals[latIdx]){
			locs <- rbind(locs, c(j,i));
		}
	}
	
	N<-length(locs[,1]); #number of locations. 
	
	# read the temperature data into the observation array.
	Temp.Vals <- matrix(NA, length(Temperature[1,1,]), N);
	latpluslon <- 1;
	for (i in seq(1,length(lonIdx))[lonIdx]){
		for (j in seq(1,length(latIdx))[latIdx]){
			Temp.Vals[,latpluslon] <- Temperature[i, j,];
			latpluslon <- latpluslon +1;
		}
	}
#	AvgTemp <- matrix(NA, 0, N);
	Temp.Vals[which(Temp.Vals==-99.9)]=NA;
	
	AvgTemp <- colMeans(Temp.Vals[c(1,2), ], na.rm=TRUE);
	for (thisseason in seq(0, seasons-2)){
		AvgTemp <- rbind(AvgTemp, colMeans(Temp.Vals[2+thisseason*3+c(1,2,3),], na.rm = TRUE));
	}

	timeline <- StartTime+seq(0, (seasons-1)/4, by=.25);
	
	returnlist <- list(locs, AvgTemp, timeline);
	names(returnlist) <- c("locations", "SeasonAvg", "Timline"); 
	return (returnlist)
}

AvgMonthTempsEurope <- function( TempData, monthvec, coordinates, StartTime=1850){
	library("ncdf")
	#TempData <- "~/Johannes/Giessen/Data/HadCRU/HadCRUT3v.nc";
	
	# Which months shall be used?
	
	# "Ja Fe Ma Ap My Jn Jl Au Se Oc No De"
	# or ordinal number 1-12
	#monthvec <- c("Jn", "Jl", "Au", "Se");
	
	# Read all the data
	NCDF.DataFile <- open.ncdf(TempData);
	Temperature <- get.var.ncdf(NCDF.DataFile);
	close.ncdf(NCDF.DataFile);

	lonW <- coordinates$lonW
	lonE <- coordinates$lonE;
	latS <- coordinates$latS;
	latN <- coordinates$latN;
	lonIdx <- (NCDF.DataFile$dim[[1]]$vals >= 360+lonW) | (NCDF.DataFile$dim[[1]]$vals <= lonE)
	latIdx <- (NCDF.DataFile$dim[[2]]$vals >= latS) & (NCDF.DataFile$dim[[2]]$vals <= latN)
	
	locs<-matrix(NA, 0, 2);
	for (i in NCDF.DataFile$dim[[1]]$vals[lonIdx]){
		for (j in  NCDF.DataFile$dim[[2]]$vals[latIdx]){
			locs <- rbind(locs, c(j,i));
		}
	}
	
	
	N<-length(locs[,1]); #number of locations. 
	
	allmonth <- c("Ja", "Fe", "Ma", "Ap", "My", "Jn", "Jl", "Au", "Se", "Oc", "No", "De")
	dummy <- matrix(NA, length(monthvec),1);
	if ( !is.integer(monthvec) ){
		for ( mon in seq(1,length(monthvec)) ){
			dummy[mon] <- grep(monthvec[mon], allmonth)
		}
	}
	monthvec <- dummy;
	years <- floor(length(Temperature[1,1,])/12);
	
	# read the temperature data into the observation array.
	Temp.Vals <- matrix(NA, length(Temperature[1,1,]), N);
	latpluslon <- 1;
	for (i in seq(1,length(lonIdx))[lonIdx]){
		for (j in seq(1,length(latIdx))[latIdx]){
			Temp.Vals[,latpluslon] <- Temperature[i, j,];
			latpluslon <- latpluslon +1;
		}
	}
	
	AvgTemp <- matrix(NA, 0, N);
	
	for (thisyear in seq(0, years-1)){
		AvgTemp <- rbind(AvgTemp, colMeans(Temp.Vals[thisyear*12 + monthvec,], na.rm = TRUE));
	}
	
	returnlist <- list(locs, AvgTemp);
	names(returnlist) <- c("locations", "AvgTemp"); 
	return (returnlist)
}

MonthTempsEurope <- function( TempData, monthvec, coordinates){
	library("ncdf")
	#TempData <- "~/Johannes/Giessen/Data/HadCRU/HadCRUT3v.nc";
	
	# Then, create a grid. Let's assume a regular grid on the lat-lon grid.
	# Set the grid distance and corner coordinates in degrees north and deg east
	# To make things easier: use grid as HadCRUT3
	lonW <- coordinates$lonW
	lonE <- coordinates$lonE;
	latS <- coordinates$latS;
	latN <- coordinates$latN;
	Delta <- 5;
	
	# Which months shall be used?
	
	# "Ja Fe Ma Ap My Jn Jl Au Se Oc No De"
	# or ordinal number 1-12
	#monthvec <- c("Jn", "Jl", "Au", "Se");
	
	# Read all the data
	TDataFile <- open.ncdf(TempData);
	Temperature <- get.var.ncdf(TDataFile);
	close.ncdf(TDataFile);
	
	locs <- NCDF.Locations( NCDF.DataFile, coordinates);
	
	N<-length(locs[,1]); #number of locations. 
	
	allmonth <- c("Ja", "Fe", "Ma", "Ap", "My", "Jn", "Jl", "Au", "Se", "Oc", "No", "De")
	dummy <- matrix(NA, length(monthvec),1);
	if ( !is.integer(monthvec) ){
		for ( mon in seq(1,length(monthvec)) ){
			dummy[mon] <- grep(monthvec[mon], allmonth)
		}
	}
	monthvec <- dummy;
	years <- floor(length(Temperature[1,1,])/12);
	
	# read the temperature data into the observation array.
	Temp.Vals <- matrix(NA, length(Temperature[1,1,]), N);
	latpluslon <- 1;
	for (i in seq(1,length(lonIdx))[lonIdx]){
		for (j in seq(1,length(latIdx))[latIdx]){
			Temp.Vals[,latpluslon] <- Temperature[i, j,];
			latpluslon <- latpluslon +1;
		}
	}
	
	MonTemp <- matrix(NA, 0, N);
	
	for (thisyear in seq(0, years-1)){
		MonTemp <- rbind(MonTemp, Temp.Vals[thisyear*12 + monthvec,]);
	}
	
	returnlist <- list(locs, MonTemp);
	names(returnlist) <- c("locations", "MonTemp"); 
	return (returnlist)
}
