# Author: Joe Werner
# Version: 0.2
# Date: 2011-12-01
#
# This program reads the temperature arrays from the huge number of
# files that is created by each simulation run. The file name is 
# assumed to be constant with a suffix "_(stepnumber).R"
#
# It requires the indices of the locations over which it has to average.
# The indices used have to be taken care of by the function or person
# calling this program.
#
# Now included are also functions that do calculations of weighted means.
# They require a vector of weights to be passed to the function of length
# equal to that of FIdx (field indices).
#
GetWeightedFieldMeans <- function( FIdx, Weights, path="./", basename="step"){
  # FIdx	: field indices, array of integer
  # path	: path to the files containing the field draws
  # basename	: the base name of the files containing the draws of the
  # 		  field
  setwd( path);
  FileList <- dir(pattern=paste(basename,"[2-5][0-9][0-9][0-9].R",sep="_"));
  NFile <- length(FileList);
  load(FileList[1]);
  Nyear <- dim(Temperature.MCMC.Sampler)[1]
  NLocs <- length(FIdx);
  
  FieldDist <- matrix(NA, nrow=Nyear, ncol=NFile)
  if ( NLocs > 1){
    FieldDist[,1] <- colSums( t( Temperature.MCMC.Sampler[, FIdx]) * Weights) / sum(Weights)

    for (DatIdx in seq(2, NFile)){
      load(FileList[DatIdx]);
      FieldDist[,DatIdx] <- colSums( t( Temperature.MCMC.Sampler[, FIdx]) * Weights) / sum(Weights)
    }
  } else {
    FieldDist[,1] <- ( Temperature.MCMC.Sampler[, FIdx])

    for (DatIdx in seq(2, NFile)){
      load(FileList[DatIdx]);
      FieldDist[,DatIdx] <- ( Temperature.MCMC.Sampler[, FIdx])
    }
  }
  return(FieldDist)
}

GetFieldMeans <- function( FIdx, path="./", basename="step"){
  # FIdx	: field indices, array of integer
  # path	: path to the files containing the field draws
  # basename	: the base name of the files containing the draws of the
  # 		  field
  setwd( path);
  FileList <- dir(pattern=paste(basename,"[2-5][0-9][0-9][0-9].R",sep="_"));
  NFile <- length(FileList);
  load(FileList[1]);
  Nyear <- dim(Temperature.MCMC.Sampler)[1]
  NLocs <- length(FIdx);
  
  FieldDist <- matrix(NA, nrow=Nyear, ncol=NFile)
  if ( NLocs > 1){
    FieldDist[,1] <- rowMeans( Temperature.MCMC.Sampler[, FIdx])

    for (DatIdx in seq(2, NFile)){
      load(FileList[DatIdx]);
      FieldDist[,DatIdx] <- rowMeans( Temperature.MCMC.Sampler[, FIdx])
    }
  } else {
    FieldDist[,1] <- ( Temperature.MCMC.Sampler[, FIdx])

    for (DatIdx in seq(2, NFile)){
      load(FileList[DatIdx]);
      FieldDist[,DatIdx] <- ( Temperature.MCMC.Sampler[, FIdx])
    }
  }
  return(FieldDist)
}

# This example plots the mean of a field (or field average) plus the 90%
# coverage (NOT the POINTWISE coverage)
#
PlotField_Quantile <- function( FIdx, path="./", basename="step"){
  FieldDist <- GetFieldMeans( FIdx, path, basename="step")
  plot(FieldDist[,1],cex=0)
  for ( a in seq(1, length(FieldDist[1,]), by=1.1)){ lines(FieldDist[,a], lwd=.1)}
  points(rowMeans(FieldDist), pch="x")
}

PlotTotalAreaMean <- function( path="./", basename="step"){
  setwd( path);
  FileList <- dir(pattern=paste(basename,"[2-5][0-9][0-9][0-9].R",sep="_"));
  load(FileList[1]);
  Locs <- c(seq(1,length(Temperature.MCMC.Sampler[1,])))
  PlotField_Quantile(Locs, path="./", basename);
}

GetAreaPointwiseMeans <- function( path="./", basename="step"){
  setwd( path);
  FileList <- dir(pattern=paste(basename,"[2-5][0-9][0-9][0-9].R",sep="_"));
  load(FileList[1]);
  NYear <- length(Temperature.MCMC.Sampler[,1])
  NLoc <- length(Temperature.MCMC.Sampler[1,])
  AreaMeans <- matrix(NA, nrow=NYear, ncol=NLoc);
  for (LocIdx in seq(1, NLoc)){
    AreaMeans[, LocIdx] <- rowMeans(GetFieldMeans( LocIdx, path, basename) );
  }
  return(AreaMeans);
}

GetAllAreaMean <- function( path="~/work/Giessen/Data/", pathpattern="Smerdon_Europe"){
  setwd(path)
  DirList <- dir(pattern=pathpattern);
  for ( DIdx in seq(1, length(DirList)) ){
    setwd(DirList[ DIdx])
    FileList <- dir(pattern=paste("step","[2-5][0-9][0-9][0-9].R",sep="_"));
    load(FileList[1]);
    Locs <- c(seq(1,length(Temperature.MCMC.Sampler[1,])))
    AreaMean <- GetFieldMeans( Locs);
    save(AreaMean, file=paste(DirList[DIdx],"AreaMean.R", sep="_") )
    setwd("..")
  }
}

