# This file contains tthe neccessary funtions to update the age depth model for
# bio-geo-chemical proxies (or in general all that have time uncertainty). Proxy
# data of this type will be put into a special list structure to reflect these
# properties.
#
# ProxADM <- list( data, adm, admIdx, CPROX)
#    data  M x 1 matrix with all measurments
#    adm   M x N matrix with N realisations of Age Depth Models
#    admIdx  which one of the N adms to choose
#    CPROX   currently valid CPROX parameters
#            FOR THE TIME BEING: beta_2 := admIdx
#
# HEADER
#
# get_agedepth ( filname)
#    read an age-depth model from the input file
#
# make_timeval ( AgeDepthModel, ProxyData, Data.timeline)
#    - take an age-depth-model and the time line
#    - search for the time values corr. to time line
#    - put proxy data at corresponding date
#
# ADM_Updater ( Pnum, ADM)
#    take the given ADM ensemble of proxy Pnum and
#    - create probability distribution for the posterior P(T | Obs, Clim)
#      where T = Time Model, Obs, Clim = Observations and drawn climate state
#    - get a random ADM from that distribution and update the Current Pars

get_agedepth <- function( filename){}

make_timeval <- function( AgeDepthModel, ProxyData, Data.timeline){
  admdup1 <- which(duplicated(AgeDepthModel))
  admdup2 <- which(duplicated(AgeDepthModel,fromLast=TRUE))
  AgeDepthModel[c(admdup1,admdup2)] <- NA
  Prox <- matrix(-99, length(Data.timeline), 1)
  TIdx <- is.element( Data.timeline, AgeDepthModel)
  TIdxA <- is.element( AgeDepthModel, Data.timeline)
  Prox[ TIdx,1] <- ProxyData[ TIdxA]
  return( Prox)
}

ADM_Updater_random <- function( Pnum, ADM, inverseTemp = 1){
    # relies on global variables:
    # Data.external
    # N.Locs
    # CProx
    pIdx <- which( colSums( Data.external[,(Pnum*N.Locs+1):((Pnum+1)*N.Locs)] != -99) > 0)[1]
    ADMnum <- dim(ADM$adm)[2]

    ProxDataADM <- (apply( ADM$adm, 2, make_timeval, ProxyData=ADM$data,
      Data.timeline = BARCAST.INPUT$Data.timeline) )
    ProxDataADM[ ProxDataADM == -99 ] <- NA
    admIdx <- sample( ADMnum,1)
    CProx[ Pnum, 5] <- admIdx
    M.InstProx[Pnum+1] <- sum(!is.na( ProxDataADM[, admIdx]))
    ProxyADM <- make_timeval( ADM$adm[,admIdx], ADM$data, BARCAST.INPUT$Data.timeline)
    out <- .Fortran( "updateSglProx", ProxyADM, as.integer( Pnum), as.integer( pIdx))
    out <- .Fortran("Tau2P_Updater", unlist(PRIORS[[6+Pnum]][2]),
      as.integer(Pnum), M.InstProx[Pnum+1]);
    return( CProx[ Pnum, ])
}

ADM_Updater <- function( Pnum, ADM, inverseTemp = 1){
    # relies on global variables:
    # Data.external
    # N.Locs
    # CProx
    pIdx <- which( colSums( Data.external[,(Pnum*N.Locs+1):((Pnum+1)*N.Locs)] != -99) > 0)[1]
    ADMnum <- dim(ADM$adm)[2]
    ProxDataADM <- (apply( ADM$adm, 2, make_timeval, ProxyData=ADM$data,
      Data.timeline = BARCAST.INPUT$Data.timeline) )
    ProxDataADM[ ProxDataADM == -99 ] <- NA
    TP <- (ProxDataADM - CProx[ Pnum,3] - CProx[ Pnum,4] * Temperature.MCMC.Sampler[-1,pIdx])^2
    Tau.List <- colSums( TP,na.rm=TRUE) * 1/(2 * CProx[ Pnum, 2]) * inverseTemp
    invProb <- Tau.List
    for( admIdx in seq(ADMnum)){
      invProb[admIdx] <- sum( exp( - Tau.List + invProb[admIdx] ))
    }
    CumProbVec <- c(0, cumsum(1/invProb)/sum(1/invProb) )
    dummy <- runif( 1, 0, 1 - .Machine$double.eps)
    admIdx <- findInterval( dummy, CumProbVec)
    CProx[ Pnum, 5] <- admIdx
    M.InstProx[Pnum+1] <- sum(!is.na( ProxDataADM[, admIdx]))
    ProxyADM <- make_timeval( ADM$adm[,admIdx], ADM$data, BARCAST.INPUT$Data.timeline)
    out <- .Fortran( "updateSglProx", ProxyADM, as.integer( Pnum), as.integer( pIdx))
    out <- .Fortran("Tau2P_Updater", unlist(PRIORS[[6+Pnum]][2]),
      as.integer(Pnum), M.InstProx[Pnum+1]);
    return( CProx[ Pnum, ])
}

ADMLogLikelihood <- function( Pnum, ADM){
    pIdx <- which( colSums( Data.external[,(Pnum*N.Locs+1):((Pnum+1)*N.Locs)] != -99) > 0)[1]
    admIdx <- CProx[ Pnum, 5]
    ProxyADM <- make_timeval( ADM$adm[,admIdx], ADM$data, BARCAST.INPUT$Data.timeline)
    ProxyADM[ ProxyADM == -99 ] <- NA
    TP <- (ProxyADM - CProx[ Pnum,3] - CProx[ Pnum,4] * Temperature.MCMC.Sampler[-1,pIdx])^2
    ADMLogL <- -1* sum( TP,na.rm=TRUE) * 1/(2 * CProx[ Pnum, 2])
    return( ADMLogL)
}


