#   BARCAST: Bayesian Algorithm to Reconstruc Climate in Space and Time
#   Copyright (C) 2010 Martin Tingley
#   Copyright (C) 2015 Johannes Werner (R adaption + parallel tempering)
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <http://www.gnu.org/licenses/>.
###############################################################################

###############################################################################

BARCAST.wd <- getwd();

source("../helper_programs/Common.R")

###############################################################################
# 
# In case R exits unexpectedly, have it automatically clean up
# resources taken up by Rmpi (slaves, memory, etc...)
.Last <- function(){
  if (is.loaded("mpi_initialize")){ 
    if (mpi.comm.size(1) > 0){ 
      print("Please use mpi.close.Rslaves() to close slaves.") 
      mpi.close.Rslaves() 
    } 
    print("Please use mpi.quit() to quit R") 
    .Call("mpi_finalize") 
  }
}

# The wrapper function
Run.Barcast <- function(BARCAST.datdir, BARCAST.infilename, pre.Sampler.Its = 1000, Sampler.Its = 5000, swap.interval=100, nChains=3, CurrCont=FALSE, invTemp.min = .01){

  # Problem: the code will now no longer really work that well on systems with very
  # few cores. Before, the code could run even on a single core machine, now it wants
  # as many as nChains slaves. Since the goal of this is to get Metropolis Coupled
  # Gibbs sampling we cannot really run the chains at different times, they need to
  # run concurrently!
  #
  # Also note: The directory structure needs to match across all slaves. This is
  # unproblematic when on a single machine, but MPI does indeed enable us to work
  # on multiple machines at the same time! Then, the data collection will be more
  # difficult.

  require(Rmpi)
  require(coda)
  # attention, this does *not* check if we have that many cores!
  mpi.spawn.Rslaves(nslaves = nChains)
  currentthin <- 1
  
  # In case we want to run this unheated (like when no AMDs are available)
  if( invTemp.min == 1) swap.interval <- Sampler.Its + 1

  # I moved the Priors file to the data directory which does indeed make sense:
  # It'll complain (well, grind to a halt) if no PriorPars.RScript file sits
  # there. I decided against putting the priors in the BARCAST.INPUT-file,
  # this way they are much more readily accessible and changeable.
  # See the PriorParsvNewModel.R file on how they need to look like.
  source(paste(BARCAST.datdir, "PriorPars.RScript", sep="/") )

  #Send the required data to slaves
  mpi.bcast.Robj2slave(BARCAST.wd)
  mpi.bcast.Robj2slave(BARCAST.datdir)
  mpi.bcast.Robj2slave(BARCAST.infilename)
  mpi.bcast.Robj2slave(pre.Sampler.Its)
  mpi.bcast.Robj2slave(Sampler.Its)
  mpi.bcast.Robj2slave(swap.interval)
  mpi.bcast.Robj2slave(Prior.Pars)
  mpi.bcast.Robj2slave(invTemp.min)
  mpi.bcast.Robj2slave(Barcast.Code_Rmpi)
  
  # Careful: The code no longer cares about automatically checking for convergence, 
  # since it is impossible to do that in a convenient way for parallel tempering.
  if( !CurrCont ) {
    Parameters.Chains <- do.call(rbind, mpi.remote.exec( Barcast.Code_Rmpi(BARCAST.datdir, 
              BARCAST.infilename,pre.Sampler.Its, Sampler.Its, swap.interval,thin=1, 
              continue=FALSE, invTemp.min = invTemp.min) ) )
    ReducedPars <- EvalParameters( Parameters.Chains, nChains, targetDir = BARCAST.datdir)
    numPar <- length(ReducedPars[, 1])/nChains

    Rhat <- foreach ( para = seq(1, numPar ), .combine="c") %dopar% {
      CheckConvergence(t(ReducedPars[seq(para, length(ReducedPars[, 1]), numPar ),] ), nstart=length(ReducedPars[1,])/3)
    }
    print(Rhat)
    save(list=c("Parameters.Chains","ReducedPars","Rhat"), file=paste(BARCAST.datdir,"/Parameters_",currentthin,"_thin.RData",sep=""))
  } else { 
    load(paste(BARCAST.datdir,"/Parameters_",currentthin,"_thin.RData",sep="")) 
    Rhat <- Rhat * 2
  }

  # Tell all slaves to close down, and exit the program
  mpi.close.Rslaves()
}

Barcast.Code_Rmpi <- function(BARCAST.datdir, BARCAST.infilename, 
	pre.Sampler.Its = 1000, Sampler.Its = 5000, swap.interval = 100, thin = 1,
	continue = FALSE, invTemp.min = .01){
  # Number of iterations of the complete sampler  Sampler.Its <- 5000;
  # Number of times to update only the temperature array before beginning to
  # update the other parameters  pre.Sampler.Its <- 1000;
  # Reduce the number of points by this factor: thin <- 1
  setwd(BARCAST.wd);
  rank <- mpi.comm.rank()
  # Load the libraries

  library("stats")
  library("MASS")

  source("../helper_programs/Common.R")
  source("CovariancePatterns.R")
  source("Proxy_Time_Updater.R")
  
  BARCAST.datdir.rank <- paste(BARCAST.datdir,rank,sep="/")
  print( BARCAST.datdir.rank)
  if(!is.null(attr(try( setwd(BARCAST.datdir.rank), silent=TRUE),"condition"))){
    dir.create( BARCAST.datdir.rank, recursive = TRUE);
    file.copy(paste(BARCAST.datdir,BARCAST.infilename,sep="/"),paste(BARCAST.datdir.rank,BARCAST.infilename,sep="/"))
  }
  BARCAST.datdir <- BARCAST.datdir.rank
  setwd(BARCAST.wd);
  BARCAST.infile <- paste ( BARCAST.datdir, "/", BARCAST.infilename , sep="");
  print(BARCAST.infile);
  BARCAST.TempFile <- paste ( BARCAST.datdir, "/Temperature.dat", sep="");
  BARCAST.CurrentFile <- paste ( BARCAST.datdir, "/BarcastCurrentRun", BARCAST.infilename, sep="");
  BARCAST.InitialFile <- paste ( BARCAST.datdir, "/BarcastInitialValues.R", sep="");

  # Load all the initial values
  #
  dyn.load("T_Updater.so")
  if ( continue){
    load(BARCAST.CurrentFile)
    print ("loaded old state")
  } else {
    # set the priors and the inital values for the MCMC sampler
    source("InitialparvalsvNewModel.R", local=TRUE)
    source("BARCAST_SetInitialValues.R",local=TRUE);

    NPat <- dim(U.Patterns)[1];
    Data.external <- Data.ALL
    Data.external[which(is.na(Data.external))] <- -99
    save(list=ls(), file= paste( BARCAST.datdir, "/InitialState.R", sep=""))
  }
  out <- .Fortran("init_updater", t(Temperature.MCMC.Sampler), CProx,
    t(Data.external), 
    t(U.Patterns), as.integer(Pattern.by.Year), as.integer(NPat),
    CURRENT.inv.spatial.corr.mat,
    as.integer(N.Times), as.integer(N.Locs), as.integer(N.PT),
    as.integer( abs( rnorm(1,0,1000))))

  ## In an attempt to speed convergence of the variance paramters
  # we will uptate only the true temperature array for a number of
  # iterations, and then add the updating of the other parameters. This is to
  # prevent the model from requiring large variances to fit the observations
  # to the data.
  
  if ( continue ){
    print("Skipping PreSampler")
  } else {
    source("BARCAST_PreSampler.R",local=TRUE);
    save(list=ls(), file=paste(BARCAST.datdir, "PreSamplerComplete.R", sep="/"))
  }
  
  source("BARCAST_PreSwitchSampler.R",local=TRUE);
  source("BARCAST_Sampler_MC3.R",local=TRUE);

  .Fortran("cleanup_updater")
  dyn.unload("T_Updater.so")
  rm(list=c("continue", "thin"))
  save(list = ls(all=TRUE), file = BARCAST.CurrentFile)

  return(Parameters.MCMC.Samples);
}

