# Evaluate the PPE, using the measures used in Werner and Tingley 2014
# Clim. Past Discuss., 10, 4499-4533, 2014
# www.clim-past-discuss.net/10/4499/2014/
# doi:10.5194/cpd-10-4499-2014

source("ScoringRules.R")

# Load target and reconstruction data
load("TrueTempsv1.R")
load("BARCASTINPUT_full_true.RData")
load("nRecon.RData")

switchIdx <- which( diff( TempSort[1,]) != 0 ) + 1

# after swapping the temperatures we need some time to settle down
WaitTime <- .6* min(unique(diff(switchIdx)))

nIter  <- dim(nRecon)[3]
nLocs  <- dim(nRecon)[2]
nYears <- dim(nRecon)[1]

# discard the first part if the chains
startIdx <- .6*nIter
thinIt   <- 10

nVals <- sum(diff(switchIdx[switchIdx > startIdx])-(min(unique(diff(switchIdx)))-WaitTime))/thinIt

Recon <- array( NA, c(nYears,nLocs, nVals))
currIdx <- 0
for( thisSwitch in which(switchIdx > startIdx)[-1] ){
  currChainIdx <- seq(switchIdx[ thisSwitch - 1]+WaitTime, switchIdx[ thisSwitch],thinIt)
  currIdx <- currIdx + seq( length(currChainIdx))
  Recon[,, currIdx] <- nRecon[,,currChainIdx]
  currIdx <- max(currIdx)
}
Recon <- Recon[,,1:currIdx]
nVals <- currIdx

# Create matrices to hold the scoring results / error measures.

# RE is an improper measure
RE.Mat <- matrix(NA,dim(Recon)[2],dim(Recon)[3])
# The cross correlation is Ok, I guess
XC.Mat <- matrix(NA,dim(Recon)[2],dim(Recon)[3])
# ditto the quadratic error
ME.Mat <- matrix(NA,dim(Recon)[2],dim(Recon)[3])
# Evaluate the Coverage Rate
CovRate  <- rep(NA, dim(Recon)[2])
CovWidth <- rep(NA, dim(Recon)[2])
# and also the Continous Ranked Probability Score, both for all years
CRPS.Mat <- matrix(NA, nYears, nLocs)
# and averaged
CRPS.avg <- matrix(NA, nLocs, 3)

Target <- Temperature.Matrix

# Exclude the top of the core, the "perfect" data
TIdx <- seq(1, nYears - 150)

# now, analyse the reconstruction
for ( locIdx in seq( nLocs) ){
  print( locIdx)
  ME.Mat[ locIdx,] <- (colMeans(( Recon[TIdx, locIdx,] - Target[TIdx, locIdx] )**2) )**.5
  XC.Mat[ locIdx,] <- apply( Recon[TIdx , locIdx,],2, cor, y=Target[TIdx, locIdx], use="pair")
  CovRate[ locIdx] <- CoverageRate( Recon[ TIdx,locIdx,], Target[ TIdx,locIdx], probs=.9)
  CovWidth[ locIdx] <- mean(diff( apply(Recon[ TIdx, locIdx,],1,quantile,probs=c(.05,.95))))
  for( thisTIdx in TIdx){
    CRPS.Mat[ thisTIdx, locIdx] <- CRPS( Recon[ thisTIdx,locIdx,], Target[thisTIdx,locIdx]) 
  }
  CRPS.avg[ locIdx, ] <- unlist(avgCRPS( Recon[ TIdx,locIdx,], Target[ TIdx,locIdx]) )
}

ErrMeas.df <- data.frame( x=BARCAST.INPUT$Master.Locs[,1], y=BARCAST.INPUT$Master.Locs[,2], xc = rowMeans( XC.Mat), rmse = rowMeans( ME.Mat), CovR90=CovRate, avgCRPS=CRPS.avg[,1], potCRPS=CRPS.avg[,2], Reli=CRPS.avg[,3], Cov90w=CovWidth)

pIdx <- c()
for( thisProx in grep( "Prox.Data", names(BARCAST.INPUT))){
  pIdx <- c( pIdx, which( colSums(!is.na(BARCAST.INPUT[[thisProx]])) > 0))
}

save(list=c("pIdx","ErrMeas.df"),file="ErrMeas.df.RData")

# make boxplots of all error measures
pdf( file="Score_Boxplot.pdf",width=8,height=6)
par( mfrow=c(2,3))
for( thisMeas in 3:dim( ErrMeas.df)[2]){
  boxplot( ErrMeas.df[,thisMeas], main=colnames(ErrMeas.df[thisMeas]))
  points( rep(1,length(pIdx)),ErrMeas.df[ pIdx, thisMeas], pch="x", col=2)
}
dev.off()

require(ggplot2)


# This can be plot using ggplot:
# par(mfrow=c(2,2))
svg(filename="Maps_%03d.svg", width=5,height=4)
  ggplot( data=ErrMeas.df) + geom_tile( aes(x,y, fill=xc) ) + scale_fill_gradient2(low="blue4",high="darkred",mid="white",limits=c(-.5,.5))
  ggplot( data=ErrMeas.df) + geom_tile( aes(x,y, fill=rmse) ) + scale_fill_gradient(low="green", high="darkmagenta")
  ggplot( data=ErrMeas.df) + geom_tile( aes(x,y, fill=Reli) ) + scale_fill_gradient(low="green", high="darkmagenta")
  ggplot( data=ErrMeas.df) + geom_tile( aes(x,y, fill=potCRPS) ) + scale_fill_gradient(low="white", high="black")
# ggplot( data=ErrMeas.df) + geom_tile( aes(x,y, fill=avgCRPS) ) + scale_fill_gradient(low="white", high="black")
dev.off()

# Analyse a low pass filtered version of the data
require(signal)

bf1.10 <- butter(1, 1/10, "low")
bf3.100 <- butter(3, 1/100, "low")
Temp10y <- apply(Temperature.Matrix, 2, filter, filt=bf1.10)
Temp100y <- apply(Temperature.Matrix, 2, filter, filt=bf3.100)
Recon10y <- NA * Recon
Recon100y <- NA * Recon
Recon10y.Q <- array(NA, c(nYears, nLocs, 9))
Recon100y.Q <- array(NA, c(nYears, nLocs, 9))
for( a in 1:nLocs){
  Recon10y[,a,] <- apply( Recon[,a,], 2, filter, filt=bf1.10)
  Recon100y[,a,] <- apply( Recon[,a,], 2, filter, filt=bf3.100)
  Recon10y.Q[,a,] <- t( apply( Recon10y[,a,], 1, quantile, probs=c(.025,.05,.1,.25,.5,.75,.9,.95,.975) ) )
  Recon100y.Q[,a,] <- t( apply( Recon100y[,a,], 1, quantile, probs=c(.025,.05,.1,.25,.5,.75,.9,.95,.975) ) )
}

Time <- seq( nYears)-1
pdf(width=12,height=5,file="Lowpass.pdf")
for( thisProx in pIdx){
  plot( Time, Temp10y[,thisProx], col="red", type="l")
  for( a in 1:nVals){lines( Time, Recon10y[,thisProx,a], col=rgb(0,0,1,.01) )}
  lines( Time, Temp10y[,thisProx], col="red")
  plot( Time, Temp10y[,thisProx], col=NA, type="l")
  for( a in 1:((dim(Recon10y.Q)[3]-1)/2) ){
      polygon( c(Time, rev(Time)), c(Recon10y.Q[,thisProx,a],rev(Recon10y.Q[,thisProx,dim(Recon10y.Q)[3]-a+1])), col=rgb(0,0,1,.2), border=NA )
  }
  lines( Temp10y[,thisProx], col="red")

  plot( Temp100y[,thisProx], col="red", type="l")
  for( a in 1:nVals){lines( Recon100y[,thisProx,a], col=rgb(0,0,1,.01) )}
  lines( Temp100y[,thisProx], col="red")
  plot( Time, Temp100y[,thisProx], col=NA, type="l")
  for( a in 1:((dim(Recon100y.Q)[3]-1)/2) ){
      polygon( c(Time, rev(Time)), c(Recon100y.Q[,thisProx,a],rev(Recon100y.Q[,thisProx,dim(Recon10y.Q)[3]-a+1])), col=rgb(0,0,1,.2), border=NA )
  }
  lines( Temp100y[,thisProx], col="red")
}
dev.off()

