# Set of often used functions for BARCAST
# 
# Author: Joe Werner
###############################################################################

require("Matrix")

EarthDistances<-function(llPoints){
	# function Dists=EarthDistances(llPoints)
	#Input is N by 2, each row a (long, lat) pair, -180<long<180; -90<lat<90.
	#Output is a N by N matrix of great circle distances in KM (approximating the
	#earth as a sphere), where the (i,j) entry is the distance between the ith
	#and jth rows of the input vector. So the diagonal is zero. 
	#This makes use of the so-called haversine formulation (see wikipedia),
	#which is also used in the m_lldist.m code of the m_map package. (m_lldist
	#gives identical results, but doesn't seem well set up for the formulation
	#of the matrix we want here.)
	#radius of the earth is taken as 6378.137
	
	RR<-6378.137; #radius of the earth in km
	NN<-length(llPoints[,1]);
	#make a NN^2 by 4 matrix. Each row one of the NN^2 possible sets of
	#two points:
	
	Pts.paired.Vec<-matrix(c(kronecker(llPoints, matrix(1,NN,1)), kronecker(matrix(1,NN,1), llPoints)),NN*NN);
	Dists.GC.AsVec<-RR*2*asin(sqrt(sin((Pts.paired.Vec[,2]-Pts.paired.Vec[,4])*pi/180/2)^2 + cos(Pts.paired.Vec[,2]*pi/180)*cos(Pts.paired.Vec[,4]*pi/180)*sin(abs(Pts.paired.Vec[,1]-Pts.paired.Vec[,3])*pi/180/2)^2));
	matrix(Dists.GC.AsVec, NN, NN)
}


matroot <- function( A){
	ev <- eigen(A);
	sqrtA <- (ev$vectors) %*% diag(sqrt(ev$values)) %*% t(ev$vectors);
	return (sqrtA);
}

LDL.decomp <- function(A){
	# returns upper unit triagonal matrix U and diagonal matrix D
	# A = U^T %*% D %*% U
	N <- length(A[1,]);
	C1 <- Cholesky( as(A, "dsCMatrix") );
	C1mat <- matrix(0, N, N);
	for (i in seq(1,N) ){
		C1mat[C1@i[i]+seq(1,C1@nz[i]),i] <- C1@x[ C1@p[i] + seq(1,C1@nz[i])]
	}
	D <- diag(diag(C1mat));
	L <- C1mat - D + diag(N);
	rlist <- list();
	rlist[[1]] <- D;
	rlist[[2]] <- L;
	names(rlist) <- c("D", "U");
	return(rlist);
}

rnewgamma <- function(n, location=1, shape, scale=1){
	# m -	location
	# b	-	scale > 0
	# a -	shape > 0
	
	if (shape < 1){
	# If a<1, one can use GAMMA(a)=GAMMA(1+a)*UNIFORM(0,1)^(1/a);
		out <- location + scale*gamma(1+scale) * runif(n, 0, 1)^(1/shape)                         
	} else {
		d <- shape - 1/3;
		c <- 1/sqrt(9*d);
		x <- rnorm(n);
		v <- 1 + c * x;
		
		indxs <- seq(1,n)[v <= 0];
		while ( length(indxs) > 0){
			indxSize <- length(indxs);
			xNew <- rnorm(indxSize);
			vNew <- shape + c * xNew;
			l <- seq(1,indxSize)[vNew > 0]
			v[indxs[l]] <- vNew[l];
			x[indxs[l]] <- xNew[l];
			indxs <- indxs[-l];
		}
		u <- runif(n);
		v <- v^3;
		x2 <- x^2;
		out <- d * v;
		
		indxs <- seq(1,n)[ (u >= 1-0.0331*x2^2) & (log(u)>=0.5*x2+d*(1-v+log(v)))];
		
		while (length(indxs) > 0){
			indxsSize <- length( indxs );
			x <- rnorm( indxsSize);
			v <- 1 + c*x;
			indxs1 <- seq(1,length(v))[v<=0];
			
			while (length(indxs1) > 0){
				indxsSize1 <- length( indxs1 );
				xNew <- rnorm( indxsSize1 );
				vNew <- shape+c*xNew;
				
				l1 <- seq(1, length(vNew))[vNew > 0];
				v[ indxs1[l1] ] <- vNew[l1];
				x[ indxs1[l1] ] <- xNew[l1];
				indxs1 <- indxs1[ -l1 ];
			}
			
			u <- runif( indxsSize );
			v <- v * v * v;
			x2 <- x*x;
			
			l <- seq(1, length(u))[(u<1-0.0331*x2*x2) | (log(u)<0.5*x2+d*(1-v+log(v))) ];
			out[ indxs[ l ] ] <- d*v[l];
			indxs <- indxs[ -l ];
		} # while ~isempty(indxs)
		
		out <- location + scale * out;
	}
	return(out);
}

CheckConvergence <- function( SampleMatrix, nstart=0, nstop=0){
# checks for convergence of the Gibbs sampler, follows Gelman et al. (2003),
# 2nd ed., chap. 11.6 ff.
#
# Inputs:
# SampleMatrix: A matrix N x M of draws of the parameter under consideration,
#   where M is the total number of chains and N is the total number of
#   draws
# nstart:  start evaluation where?
# nstop :  end evaluation where?
# 
# Calculate the between and in-between varianced B and W and the convergence
# parameter Rhat. Returns Rhat only.
#
  if ( nstart == 0){ nstart <- 1}
  if ( nstop  == 0){ nstop  <- length(SampleMatrix[,1])}
  m <- length(SampleMatrix[1,])
  n <- nstop - nstart + 1

  psibarJ <- colMeans(SampleMatrix[nstart:nstop,])
  psibar  <- mean(psibarJ)
  B <- n/(m - 1) * sum( (psibarJ - psibar)^2)
  
  s2j <- 1/(n-1) * rowSums( ( t(SampleMatrix[nstart:nstop,]) - psibarJ)^2 )
  W <- mean( s2j)

  varhat <- (n - 1)/n * W + 1/n * B

  Rhat <- sqrt( varhat/W)

  return(Rhat)
}

EvalParameters <- function( Parameters.Chains, nChains, targetDir ="./"){
  require( coda)
  require( ggmcmc)
  
  nPar    <- dim(Parameters.Chains)[1]/nChains
  # Construct the selection of indices that are relevant for the assessment, as 
  # some of the parameters simply encode proxy type or are reseved for an
  # unimplemented (rather: unused) parameter
  SysPars <- c(1,2,3,4,6)    # process level model
  ProxPars<- c(2,3,4)        # data level model
  ProxPars_red<- c(2,3,4,5)  # data level model for "red trees"
  MCMC.Idx <- SysPars
  Prox.Idx <- grep("type", unique(rownames(Parameters.Chains)) )
  
  for( thisProx in Prox.Idx){
    if( Parameters.Chains[ thisProx] == 1) {
      MCMC.Idx <- c(MCMC.Idx, thisProx + ProxPars_red - 1)
    } else {
      MCMC.Idx <- c(MCMC.Idx, thisProx + ProxPars - 1)
    }
  }

  Parameters.MCMC <- mcmc.list()
  ReducedPars     <- matrix(NA, ncol=dim(Parameters.Chains)[2],nrow=0)
  for( pIdx in seq(nChains)){
    Parameters.MCMC[[pIdx]] <- mcmc( t(Parameters.Chains[(pIdx-1)*nPar+MCMC.Idx,]) )
    colnames( Parameters.MCMC[[pIdx]]) <- rownames( Parameters.Chains[MCMC.Idx,] )
    ReducedPars <- rbind( ReducedPars, Parameters.Chains[(pIdx-1)*nPar+MCMC.Idx,])
    rownames( ReducedPars)[ (pIdx-1)*length(MCMC.Idx)+(1:length(MCMC.Idx))] <- rownames( Parameters.Chains[MCMC.Idx,] )
  }
  Parameters.GG <- ggs(Parameters.MCMC, parallel=TRUE)

  PlotList <- c("ggs_density()", "ggs_traceplot()", "ggs_autocorrelation()",
          "ggs_crosscorrelation()", "ggs_Rhat()", "ggs_geweke()")
  ggmcmc(Parameters.GG,file=paste( targetDir,"ggmcmc.pdf",sep="/"),plot=PlotList)
  return( ReducedPars)
}

