! You need to typecast only the integers when calling the Fortran subroutines, 
! as R seems to initialise all numbers to float (well, "numeric"), and those seem
! to be handled well enough.
!
! The temperature matrix, the observations and the current inverse correlation matrix
! are saved between different function calls, they are globally defined variables
! in the module TUPDATER_DATA
!
! The CISCM needs to be written after each updater step of the parameters
!
! 2014-05-29
! Moved the proxy parameters CPROX into the main module to plan for the future.
! 2011-02-19
! Removed unnecessary cholesky decomposition at each T_t update step. It is now
! calculated when updating the Post.Cov.Now Array after parameter updates
!
MODULE TUPDATER_DATA
  
  ! One problem with variables defined in modules:
  ! It slows execution down, probably a lot, because the code needs to
  ! constantly check for changes and possible reallocations. This is a bit
  ! strange to me. This is avoided when calling from inside a module.

  INTEGER :: IX=123                             ! random seed for ziggurat
  INTEGER :: NYEARS, NLOCS, NPT                 ! #years, #locations, #proxies
  DOUBLE PRECISION, ALLOCATABLE :: T_MATRIX(:,:), DATA_ALL(:,:), Ciscm(:,:)
  ! T_MATRIX	: NLOCS x NLOCS 		current posterior of temperature
  ! DATA_ALL	: NLOCS*(NPT+1) x NYEARS 	observations / proxies
  ! Ciscm	: NLOCS x NLOCS			current inverse covariance matrix
  DOUBLE PRECISION, ALLOCATABLE :: UPatterns(:,:), CProx(:,:)
  INTEGER, ALLOCATABLE :: PatByYear(:)
  INTEGER :: NumPat

CONTAINS

SUBROUTINE getzig( DUMMY, NRAN)

  USE Ziggurat

  IMPLICIT INTEGER(I - K)
  INTEGER :: NRAN
  DOUBLE PRECISION :: DUMMY( NRAN)

  DO I = 1, NRAN
     DUMMY( I) = rnor( )
  END DO

END SUBROUTINE getzig

!------------------------------------------------------
! subroutines to draw new temperature estimates
!------------------------------------------------------
subroutine T0Updater( T0p, Cpars)

  !USE TUPDATER_DATA

  IMPLICIT INTEGER (I-K)

  double precision, intent(in) :: T0p(2), Cpars(*)

  double precision, dimension(Nlocs) :: Ones
  double precision, dimension(NLocs, NLocs) :: postIcovM, postCovMat, Eye
  integer :: INFO 
  
  postCovMat = 0.0
  do i = 1, NLocs
    postCovMat(i,i) = 1
  enddo
  postIcovM = postCovMat
  Eye = postCovMat

  ! Find the (conditional) posterior covariance matrix 
  !  (In the notation of the handwritten notes, this is Psi):
  CALL DGEMM( 'N', 'N', NLocs, NLocs, NLocs, Cpars(1)*Cpars(1)/Cpars(3), Ciscm, NLocs, &
    Eye, NLocs, 1/T0p(2), postIcovM, NLocs)

  ! now invert postCovMat to get Psi
  CALL DPOSV('U', NLocs, NLocs, postIcovM, NLocs, postCovMat, NLocs, INFO)
!  PRINT *, "The matrix inversion Error Code", INFO

  ! Calculate the mean vector:
  ! construct a vector for the residuals
  Ones = 1
  
  T_MATRIX(1:NLOCS,0) = Cpars(1)*T_MATRIX(1:NLOCS,1) - Cpars(1)*(1-Cpars(1))*Cpars(2)*Ones
  CALL DGEMV( 'N', NLocs, NLocs, 1/Cpars(3), Ciscm, NLocs, T_MATRIX(1:NLOCS,0), 1, T0p(1)/T0p(2), Ones, 1)
  T_MATRIX(1:NLOCS,0) = Ones
  CALL DGEMV( 'N', NLOCS, NLOCS, 1.0D0, postCovMat, NLOCS, Ones, 1, 0.0D0, T_MATRIX(1:NLOCS,0), 1)
  ! store the mean in T0
  ! in order to sample:
  ! draw NLocs-dimensional normally dist iid random numbers
  ! calculate Cholesky factor of postCovMat = R' %*% R
  ! then
  ! T0.new = postMean + R' %*% Y
  CALL DPOTRF('U', NLocs, postCovMat, NLocs, INFO)   ! Cholesky
  DO i = 2, NLocs
    DO j = 1, i - 1
      postCovMat(i, j) = 0.0D+00
    ENDDO
  ENDDO
  ! overwrite "Ones" by the random numbers, don't need it in this routine anymore
  CALL getzig( Ones, NLOCS)
  CALL DGEMV( 'N', NLOCS, NLOCS, 1.0D+00, postCovMat, NLOCS, Ones, 1, 1.0D+00, T_MATRIX(1:NLOCS, 0), 1)

END SUBROUTINE T0Updater

subroutine TkUpdater_sparse( TIDX, Cpars, Hk, Hkja, Hkia, PostCovNow, CholPostCovNow)

  !USE TUPDATER_DATA

  IMPLICIT INTEGER (I-K)

  integer, intent(in) :: TIDX ! The idx of the current year / season / whatever
  double precision, intent(in) :: Cpars(*)
  ! The H matrix, a stack of diagonal matrices with the factor beta_1
  double precision, intent(in) :: Hk(*)
  integer, intent(in) :: Hkia(*), Hkja(*)
  double precision, intent(in) :: PostCovNow(NLOCS, NLOCS)
  double precision, intent(in) :: CholPostCovNow(NLOCS, NLOCS)

  integer :: BIDX, EIDX ! the indices for the proxy loops
  double precision, dimension(Nlocs) :: Ones
  double precision, dimension(NLocs, NLocs) :: Eye
  double precision, allocatable :: BVec(:), Wgood(:)
  integer :: NObsTotal, HNNZ
  
  Ones = 1.0D0
  Eye= 0.0D0
  do i = 1, NLocs
    Eye(i,i) = 1.0D0
  enddo
  NObsTotal = NLOCS * (NPT + 1)
  ! Need to build:
  ! BVec
  ALLOCATE( BVec(NObsTotal) )
  ALLOCATE( Wgood(NObsTotal) )
  ! populate the matrices / vectors, remove the missing obs for this year later (mask with WHERE)
  BVec(1:NLOCS) = 0.0D+00
  DO I = 1, NPT
    BIDX = NLOCS * I + 1
    EIDX = (I+1) * NLOCS
    BVec(BIDX:EIDX) = CProx( I, 3)!9 + 3*(I-1))
    IF ( CProx( I, 1) == 1) THEN
      IF ( TIDX > 1) THEN
        BVec(BIDX:EIDX) = BVec(BIDX:EIDX) + &
             UPatterns(BIDX:EIDX,PatByYear(TIDX - 1)) * CProx( I, 5) * DATA_ALL(BIDX:EIDX, TIDX - 1)
      END IF
    END IF
  ENDDO
  T_MATRIX(1:NLOCS, TIDX) = Cpars(1)*(T_MATRIX(1:NLOCS, TIDX + 1) + T_MATRIX(1:NLOCS, TIDX -1) ) +&
    (1-Cpars(1))*(1-Cpars(1))*Cpars(2)*Ones

  WHERE( DATA_ALL(1:NObsTotal, TIDX) == -99 )
    Wgood = 0
  ELSEWHERE
    Wgood = DATA_ALL(1:NObsTotal,TIDX) - BVec(1:NObsTotal)
  END WHERE
  
  HNNZ = Hkia(NLOCS+1) - 1

  CALL amux( NLOCS, Wgood(1:NObsTotal), Ones(1:NLOCS), Hk(1:HNNZ), Hkja(1:HNNZ), Hkia(1:NLOCS+1))
  CALL DGEMV( 'N', NLOCS, NLOCS, 1.0D0/Cpars(3), Ciscm, NLOCS, T_MATRIX(1:NLOCS, TIDX), 1, 1.0D+00, Ones, 1)

  CALL DGEMV( 'N', NLOCS, NLOCS, 1.0D+00, PostCovNow, NLOCS, Ones, 1, 0.00D+00, T_MATRIX(1:NLOCS, TIDX), 1)
  CALL getzig( Ones, NLOCS)
  CALL DGEMV( 'N', NLOCS, NLOCS, 1.0D+00, CholPostCovNow, NLOCS, Ones, 1, 1.0D+00, T_MATRIX(1:NLOCS, TIDX), 1)
  
  DEALLOCATE( BVec)
  DEALLOCATE( Wgood)

END SUBROUTINE TkUpdater_sparse

subroutine TlastUpdater( Cpars, UPatNow, PostCovNow, CholPostCovNow)

  !USE TUPDATER_DATA

  IMPLICIT INTEGER (I-K)

  integer :: BIDX, EIDX ! the indices for the proxy loops
  double precision :: Cpars(*)
  double precision :: UPatNow(*)
  double precision :: PostCovNow(NLOCS, NLOCS)
  double precision :: CholPostCovNow(NLOCS, NLOCS)
  integer :: NObsI, NObsP(NPT)
  double precision, dimension(Nlocs) :: Ones
  double precision, dimension(NLocs, NLocs) :: Eye
  ! The H matrix, a stack of diagonal matrices with the factor beta_1
  double precision, allocatable :: Hk(:,:), Hdummy(:,:)
  double precision, allocatable :: ErrIcovM(:,:)
  double precision, allocatable :: BVec(:), Wgood(:)
  
  Ones = 1.0D0
  Eye= 0.0D0
  do i = 1, NLocs
    Eye(i,i) = 1.0D0
  enddo
  
  ! Need to build:
  ! Hk, ErrIcovM, BVec
  ! Find number of prox and inst obs, allocate memory
  NObsI = SUM( UPatNow(1:NLOCS) )
  DO i = 1, NPT
    NObsP(i) = SUM( UPatNow(NLOCS * i + 1: (NLOCS+1) * i) )
  ENDDO
  !NObsTotal = SUM( NObsP) + NObsI
  ! I will first do this without removing the "missing obs" lines
  NObsTotal = NLOCS * (NPT + 1)
  ALLOCATE( Hk( NObsTotal, NLOCS) )
  ALLOCATE( Hdummy( NLOCS, NObsTotal) )
  ALLOCATE( ErrIcovM( NObsTotal, NObsTotal) )
  ALLOCATE( BVec(NObsTotal) )
  ALLOCATE( Wgood(NObsTotal) )
  ! find locations w/ inst + proxy observations this year and populate the matrices
  ! using the UPatNow and the SPREAD command
  ErrIcovM = 0.0D+00
  Hk(1:NLOCS,1:NLOCS) = SPREAD(UPatNow(1:NLOCS), 1, NLOCS)
  Hk(1:NLOCS,1:NLOCS) = MERGE(Hk(1:NLOCS,1:NLOCS), ErrIcovM(1:NLOCS,1:NLOCS), Eye == 1)
  ErrIcovM(1:NLOCS,1:NLOCS) = 1.0D+00 / Cpars(6) * Eye
  BVec(1:NLOCS) = 0.0D+00
  DO I = 1, NPT
    BIDX = NLOCS * I + 1
    EIDX = (I+1) * NLOCS
    Hk(BIDX:EIDX,1:NLOCS) = SPREAD(UPatNow(BIDX:EIDX), 1, NLOCS)*CProx( I, 4)
    Hk(BIDX:EIDX,1:NLOCS) = MERGE(Hk(BIDX:EIDX,1:NLOCS), ErrIcovM(BIDX:EIDX,BIDX:EIDX), Eye == 1)
    ErrIcovM(BIDX:EIDX,BIDX:EIDX) = 1.0D+00 / CProx( I, 2) * Eye
    BVec(BIDX:EIDX) = CProx( I, 3)
  ENDDO
  T_MATRIX(1:NLOCS, NYEARS) = Cpars(1)*(T_MATRIX(1:NLOCS, NYEARS -1) ) + (1-Cpars(1))*Cpars(2)*Ones
  Wgood(1:NObsTotal) = DATA_ALL(1:NObsTotal,NYEARS) - BVec(1:NObsTotal)
  CALL DGEMM( 'T', 'N', NLOCS, NObsTotal, NObsTotal, 1.0D+00, Hk, NObsTotal, ErrIcovM, NObsTotal, 0.0D+00, &
    Hdummy, NLOCS)
  CALL DGEMV( 'N', NLOCS, NLOCS, 1.0D0/Cpars(3), Ciscm, NLOCS, T_MATRIX(1:NLOCS, NYEARS), 1, 0.0D+00, Ones, 1)
  CALL DGEMV( 'N', NLOCS, NObsTotal, 1.0D+00, Hdummy, NLOCS, Wgood, 1, 1.0D+00, Ones, 1)
  CALL DGEMV( 'N', NLOCS, NLOCS, 1.0D+00, PostCovNow, NLOCS, Ones, 1, 0.00D+00, T_MATRIX(1:NLOCS, NYEARS), 1)
  CALL getzig( Ones, NLOCS)
  CALL DGEMV( 'N', NLOCS, NLOCS, 1.0D+00, CholPostCovNow, NLOCS, Ones, 1, 1.0D+00, T_MATRIX(1:NLOCS, NYEARS), 1)
  
  DEALLOCATE( Hk)
  DEALLOCATE( Hdummy)
  DEALLOCATE( ErrIcovM)
  DEALLOCATE( BVec)
  DEALLOCATE( Wgood)

END SUBROUTINE TlastUpdater

! -----------------------------------------------------------------------------
! The full time updater step / driver for the updates
! -----------------------------------------------------------------------------
SUBROUTINE T_Updater( PriorT0, Cpars, CCovArray, UCCovArray)

  !USE TUPDATER_DATA

  IMPLICIT INTEGER (I-K)

  double precision :: PriorT0(2)
  double precision :: Cpars(*)
  double precision, dimension(1:NLOCS*(1+NPT) ) :: UPatNow
  double precision :: CCovArray(NLOCS, NLOCS, NumPat)
  double precision :: UCCovArray(NLOCS, NLOCS, NumPat)
  double precision :: CCov(NLOCS, NLOCS)
  double precision :: UCCov(NLOCS, NLOCS)
  integer :: PatIDX, NObsTotal, TotalObs

  double precision :: NObsI( NumPat), NObsP( NPT, NumPat)
  double precision, dimension(NLocs, NLocs) :: Eye
  ! The Hk matrix space will have to contain all the products Hk^T %*% Xi^-1
  ! actually this is Hk^T !
  ! First it will be populated with H for each pattern, 
  ! then it will be converted to CSR
  ! then it will be multiplied
  double precision, allocatable :: Hk(:), Hdummy(:,:)
  integer, dimension(1:(NLOCS + 1)*NumPat):: Hkia
  integer, allocatable          :: Hkja(:)
  ! begin and end indices for addressing the different submatrices of Hk and Xi
  integer :: BIDX, EIDX, HAdd( 0:NumPat)
  ! integer error locator
  integer :: IERR
  double precision, allocatable :: ErrIcovMDiag(:)

  ! create diagonal matrices to use e.g. as MASK
  Eye= 0.0D0
  do i = 1, NLocs
    Eye(i,i) = 1.0D0
  enddo
  
  NObsTotal = NLOCS * (NPT + 1)
  ! iterate over all different pattern types:
  ! Need to build the Hk and Xi^-1 matrices, convert Hk to spares representation
  ! and (finally, cf. next step) pass to TkUpdater
  ! There is no need for Xi^-1 in CSR as we can use AMUDIA or DIAMUA to multiply
  ! a diagonal matrix with a CSR one
  !
  ! Create the Xi^-1 (ErrIcovM) Matrix in diagonal representation
  ! needs additionally information on 
  !   number of diagonals (ndiag) = 1
  !   dimension of matrix (n) = NLOCS * (NPT + 1) = NObsTotal
  ! if real DIA format is needed. However, the AMUDIA subroutine can do without
  ! this information.
  ALLOCATE( ErrIcovMDiag(NObsTotal) )
  ErrIcovMDiag(1:NLOCS) = 1.0D+00 / Cpars(6)
  DO i = 1, NPT
    BIDX = NLOCS * I + 1
    EIDX = (I+1) * NLOCS
    ErrIcovMDiag(BIDX:EIDX) = 1.0D+00 / CProx( I, 2)
  ENDDO

  ! Now: 
  ! - Create Hk for all patterns
  ! - convert to CSR
  ! - multiply with Xi^-1
  TotalObs = SUM(UPatterns(1:NLOCS*(1+NPT),1:NumPat))
  ALLOCATE( Hdummy(NLOCS, NObsTotal))
  ALLOCATE( Hk( TotalObs) )
  ALLOCATE( Hkja( TotalObs))
  ! IA is #of rows*#patterns + #of patterns, at each #rows+1 the array size of 
  ! Hk and Hkja is stored

  HAdd(0) = 1
  DO J = 1, NumPat
    UPatNow = UPatterns(1:NObsTotal, J)
    ! Need to build Hk
    ! Find number of prox and inst obs
    ! Store this in the voctors NObsI and NObsP
    ! Store also in the address vector Hadd for the Hk matrix addresses
    HAdd(J) = HAdd(J-1) + INT(SUM (UPatNow) )
    NObsI( J) = SUM( UPatNow(1:NLOCS) )
    ! find locations w/ inst + proxy observations this year and populate the matrices
    ! spread creates a matrix from a vector by repeating the vector for each col (or row)
    Hdummy(1:NLOCS, 1: NLOCS) = SPREAD(UPatNow(1:NLOCS), 1, NLOCS)
    ! the next line selects the "allowed" elements on the main diagonal
    Hdummy(1:NLOCS, 1: NLOCS) = MERGE(Hdummy(1:NLOCS, 1: + NLOCS), Eye, Eye == 1)
    DO I = 1, NPT
      BIDX = NLOCS * I + 1
      EIDX = (I+1) * NLOCS
      NObsP( I, J) = SUM( UPatNow( BIDX: EIDX) )
      Hdummy(1:NLOCS, BIDX:EIDX) = SPREAD(UPatNow( BIDX: EIDX), 1, NLOCS)*CProx( I, 4)
      Hdummy(1:NLOCS, BIDX:EIDX) = MERGE(Hdummy(1:NLOCS, BIDX:EIDX), Eye, Eye == 1)
    ENDDO
    ! convert to sparse representation
    ! input is the Hdummy array, output is a portion of the Hk and Hkia, Hkja matrices
    CALL dnscsr(NLOCS, NObsTotal, Hadd(J)-Hadd(J-1), Hdummy(1:NLOCS, 1:NObsTotal), NLOCS, &
      Hk(Hadd(J-1):Hadd(J)-1), Hkja(Hadd(J-1):Hadd(J)-1), Hkia((NLOCS +1)*(J-1)+1:(NLOCS+1)*J), IErr )
    ! multiply with ErrIcovMDia using AMUDIA
    CALL amudia(NLOCS, 1, Hk(Hadd(J-1):Hadd(J)-1), Hkja(Hadd(J-1):Hadd(J)-1), Hkia((NLOCS +1)*(J-1)+1:(NLOCS+1)*J), &
      ErrIcovMDiag(1:NObsTotal), Hk(Hadd(J-1):Hadd(J)-1), Hkja(Hadd(J-1):Hadd(J)-1), Hkia((NLOCS +1)*(J-1)+1:(NLOCS+1)*J))
  ENDDO
  ! The matrix Hk should now contain all matrices HXI_k = (Hk^T %*% Xi^-1) in 
  ! subsequent blocks (columnwise), that is:
  ! Hk = (H_1^T %*% Xi^-1, H_2^T %*% Xi^-1, ... , H_NumPat^T %*% Xi^-1)
  ! They can be addressed by the PatIDX in the following loop
  
  PatIDX = PatByYear(NYEARS)
  CCov  = CCovArray( 1:NLOCS, 1:NLOCS, PatIDX)
  UCCov = UCCovArray(1:NLOCS, 1:NLOCS, PatIDX)
  CALL TlastUpdater(Cpars, UPatterns(1:NLOCS*(1+NPT),PatIDX), CCov, UCCov)
  DO I = NYEARS-1, 1, -1
    PatIDX = PatByYear(I)
    CCov  = CCovArray( 1:NLOCS, 1:NLOCS, PatIDX)
    UCCov = UCCovArray(1:NLOCS, 1:NLOCS, PatIDX)
    BIDX = Hadd(PatIDX-1) 
    EIDX = Hadd(PatIDX)-1 
    CALL TkUpdater_sparse(I, Cpars, Hk(BIDX:EIDX), Hkja(BIDX:EIDX), &
      Hkia((NLOCS +1)*(PatIDX-1)+1:(NLOCS+1)*PatIDX), CCov, UCCov)
  ENDDO
  CALL T0Updater(PriorT0, Cpars)

  DEALLOCATE( Hk)
  DEALLOCATE( ErrIcovMDiag)
  DEALLOCATE( Hkja)
  DEALLOCATE( Hdummy)


END SUBROUTINE T_Updater

END MODULE TUPDATER_DATA

!--------------------------------------------------------------------------------
! INTERFACE to the TUPDATER_DATA MODULE
!--------------------------------------------------------------------------------

SUBROUTINE TUpdater( PriorT0, Cpars, CCovArray, UCCovArray)
  ! call the Temp updater from the module
  USE TUPDATER_DATA
  
  double precision :: PriorT0(2)
  double precision :: Cpars(*)
  double precision :: CCovArray(NLOCS, NLOCS, NumPat)
  double precision :: UCCovArray(NLOCS, NLOCS, NumPat)

  CALL T_Updater( PriorT0, Cpars, CCovArray, UCCovArray)

END SUBROUTINE TUpdater

SUBROUTINE init_updater( INITIAL_T, CProxCurr, ProxData, UPat, PbyY, NPat, currinvcovmat, NT, NL, NP, ZIGSEED)
  ! initialise the data
  USE Ziggurat
  USE TUPDATER_DATA
  
  IMPLICIT NONE
  
  INTEGER :: ZIGSEED, NT, NL, NP, NPat
  DOUBLE PRECISION :: INITIAL_T( NL, NT + 1), currinvcovmat(NL, NL), ProxData(NL*(NP+1),NT)
  DOUBLE PRECISION :: UPat(1:NL*(NP+1), 1:NPat)
  INTEGER :: PbyY(1:NT)
  DOUBLE PRECISION :: CProxCurr(1:NP, 1:5)
  
  NYEARS = NT
  NLOCS = NL
  NPT = NP
  NumPat = NPat

  IF ( ZIGSEED == 0 ) ZIGSEED = 123
  CALL zigset( ZIGSEED)
  
  IF ( .NOT. ALLOCATED( T_MATRIX) ) ALLOCATE ( T_MATRIX(1 : NLOCS, 0 : NYEARS))
  IF ( .NOT. ALLOCATED( DATA_ALL) ) ALLOCATE ( DATA_ALL(1 : NLOCS*(NPT + 1), 1 : NYEARS))
  IF ( .NOT. ALLOCATED( Ciscm) ) ALLOCATE ( Ciscm(1 : NLOCS, 1 : NLOCS))
  IF ( .NOT. ALLOCATED( UPatterns) ) ALLOCATE ( UPatterns (1 : NLOCS*(NPT+1), 1:NumPat) )
  IF ( .NOT. ALLOCATED( PatByYear) ) ALLOCATE ( PatByYear( NYEARS))
  IF ( .NOT. ALLOCATED( CProx) ) ALLOCATE ( CProx(1:NP, 1:5) )

  T_MATRIX = INITIAL_T
  DATA_ALL = ProxData
  Ciscm    = currinvcovmat
  UPatterns = UPat
  PatByYear = PbyY
  CProx = CProxCurr

END SUBROUTINE init_updater

SUBROUTINE updateUPat( UPat, PbyY, NPat)
  ! populate the new pattern data sets
  USE TUPDATER_DATA
  
  IMPLICIT NONE
  
  INTEGER :: NPat
  DOUBLE PRECISION :: UPat(1:NLOCS*(NPT+1), 1:NPat)
  INTEGER :: PbyY(1:NYEARS)
  
  NumPat = NPat
  
  DEALLOCATE( UPatterns)
  IF ( .NOT. ALLOCATED( UPatterns) ) ALLOCATE ( UPatterns (1 : NLOCS*(NPT+1), 1:NumPat) )

  UPatterns = UPat
  PatByYear = PbyY

END SUBROUTINE updateUPat

SUBROUTINE updateSglProx( ProxData, PNum, PIdx)
  ! change the data of a single proxy
  USE Ziggurat
  USE TUPDATER_DATA
  
  IMPLICIT NONE
  
  INTEGER :: PNum, PIdx
  DOUBLE PRECISION :: ProxData(1: NYEARS)
  
  DATA_ALL( PNum*NLOCS + PIdx, 1: NYEARS) = ProxData

  ! PatByYear = PbyY also needs to be updated!

END SUBROUTINE updateSglProx

SUBROUTINE set_ciscm( currinvcovmat)
  ! update the inverse covariance matrix
  USE TUPDATER_DATA

  DOUBLE PRECISION :: currinvcovmat(NLOCS,NLOCS)
  Ciscm    = currinvcovmat

END SUBROUTINE set_ciscm

SUBROUTINE get_ciscm( currinvcovmat)
  ! return the inverse covariance matrix
  USE TUPDATER_DATA

  DOUBLE PRECISION :: currinvcovmat(NLOCS,NLOCS)
  currinvcovmat = Ciscm

END SUBROUTINE get_ciscm

SUBROUTINE cleanup_updater()
  ! deallocate everything
  USE TUPDATER_DATA
  
  IMPLICIT NONE
 
  DEALLOCATE ( T_MATRIX)
  DEALLOCATE ( DATA_ALL)
  DEALLOCATE ( Ciscm)
  DEALLOCATE ( UPatterns )
  DEALLOCATE ( PatByYear )

END SUBROUTINE cleanup_updater

SUBROUTINE GetTMatrix( TMat)
  !
  ! returns the current temperature matrix
  USE TUPDATER_DATA

  IMPLICIT NONE
  DOUBLE PRECISION, DIMENSION(1:NLOCS, 0:NYEARS) :: TMat

  TMat = T_MATRIX

END SUBROUTINE GetTMatrix


! -----------------------------------------------------------------------------
! 
! The parameters Theta=[alpha, mu, sigma, phi, sigma_I, sigma_P, beta_{0,1...}

SUBROUTINE Alpha_Updater( alphaprior, Cpars)
  !inputs are: [PRIORS.alpha, CURRENT_PARS ]
  !
  !#UPDATES the AR(1) coefficient alpha in the main BARCAST code
  !
  ! TRUNCATED NORMAL
  
  USE Ziggurat
  USE TUPDATER_DATA

  IMPLICIT NONE
  DOUBLE PRECISION :: TMatLessMean(1:NLOCS, 0:NYEARS)
  DOUBLE PRECISION :: AlphaMean, AlphaVar
  DOUBLE PRECISION :: Dummy( 1:NLOCS, 0:NYEARS)
  DOUBLE PRECISION :: alpha(1), alphaprior(*), Cpars(*)
  
  !calculate the inverse posterior variance.
  !first lay down the matrix of the deviatons of the temperature values from
  !the dependant means. 
  TMatLessMean(1:NLOCS, 0:NYEARS) = T_MATRIX(1:NLOCS, 0:NYEARS) - Cpars(2)
  
  CALL DGEMM('N', 'N', NLOCS, NYEARS, NLOCS, 1.0/Cpars(3), Ciscm, NLOCS, TMatLessMean(1:NLOCS,0:NYEARS-1), NLOCS, &
    0.0D0, Dummy(1:NLOCS,0:NYEARS-1), NLOCS)
 
  AlphaVar = 1.0D0/(SUM( TMatLessMean(1:NLOCS,0:NYEARS-1) * Dummy(1:NLOCS,0:NYEARS-1) ) + 1.0/alphaprior(2))

  AlphaMean = ( SUM( TMatLessMean(1:NLOCS,1:NYEARS) * Dummy(1:NLOCS,0:NYEARS-1) ) + alphaprior(1)/alphaprior(2))
  AlphaMean = AlphaVar * AlphaMean
  
  CALL getzig( alpha, 1)
  alpha = AlphaMean + alpha * AlphaVar**.5
  Cpars(1) = alpha(1)
  IF (Cpars(1) < 0) Cpars(1) = 0

END SUBROUTINE Alpha_Updater
  
SUBROUTINE Mu_Updater( muprior, Cpars)
  
  USE Ziggurat
  USE TUPDATER_DATA

  IMPLICIT NONE
  DOUBLE PRECISION :: TMatDiff(1:NLOCS, 1:NYEARS)
  DOUBLE PRECISION :: MuMean, MuVar
  DOUBLE PRECISION :: Ones_L(1:NLOCS), Ones_Y(1:NYEARS)
  DOUBLE PRECISION :: muprior(*), Cpars(*), DummyVec(1:NLOCS)

  ! function New_mu=Mu_Updater_vNM(mu_p, T_Mat, C_pars, C_iscm);
  !
  !inputs are: [PRIORS.mu, CURRENT_PARS] 
  !
  !#UPDATES the constant mean coefficient of the AR(1) process in the
  !#main BARCAST code
  !
  ! NORMAL

  Ones_Y(1:NYEARS) = 1

  MuVar = 1.0D0/(1.0D0/muprior(2) + 1.0D0/Cpars(3) * SUM(Ciscm) * NYEARS*(1-Cpars(1))**2 )
  
  !calculate the posterior mean:
  ! first the difference, T_mat less shifted (in time) and scaled (by
  ! a*N_T*(1-alpha) T_mat, then summed across time:
  TMatDiff = T_MATRIX(1:NLOCS, 1:NYEARS) - Cpars(1)*T_MATRIX(1:NLOCS, 0:NYEARS-1)
  CALL DGEMV('N', NLOCS, NYEARS, 1.0D0, TMatDiff, NLOCS, Ones_Y, 1, 0.0D0, Ones_L, 1)
  CALL DGEMV('N', NLOCS, NLOCS, 1.0D0/Cpars(3), Ciscm, NLOCS, Ones_L, 1, 0.0D0, DummyVec, 1)

  MuMean = MuVar * ( muprior(1)/muprior(2) + (1-Cpars(1)) * SUM(DummyVec)  )
  
  CALL getzig( Cpars(2), 1)
  Cpars(2) = MuMean + Cpars(2) * MuVar**.5
END SUBROUTINE Mu_Updater

SUBROUTINE Sigma2_Updater( sigma2prior, Cpars)

  USE Ziggurat
  USE random
  USE TUPDATER_DATA

  IMPLICIT NONE
  REAL :: alpha
  DOUBLE PRECISION :: sigma2prior(*), Cpars(*), beta
  DOUBLE PRECISION :: TDiffMat(1:NLOCS, 1:NYEARS), Dummy(1:NLOCS, 1:NYEARS)

  alpha = NYEARS * NLOCS /2.0 + sigma2prior(1)

  TDiffMat = T_MATRIX(1:NLOCS,1:NYEARS) - Cpars(1) * T_MATRIX(1:NLOCS,0:NYEARS - 1) - &
    (1-Cpars(1) ) * Cpars(2)
  CALL DGEMM( 'N', 'N', NLOCS, NYEARS, NLOCS, 1.0D0, Ciscm, NLOCS, TDiffMat, NLOCS, 0.0D0, Dummy, NLOCS)
  TDiffMat = TDiffMat * Dummy
  beta = sigma2prior(2) + 0.5*SUM(TDiffMat)
  !
  ! concerning Hobbits (the Inv-Gamma distribution)
  ! for a random varaible X the following holds:
  ! 1/X ~ Gamma (a, 1/b) <-> X ~ InvGamma(a, b)
  ! X ~ InvGamma (a, b) <-> k*X ~ InvGamma(a, k*b)
  ! So:
  ! Y ~ Gamma(a, 1)
  ! X = 1/Y ~ InvGamma(a, 1)
  ! sigma2 = beta*X ~ InvGamma(a, beta)
  ! sigma2 ~ beta/Y
  Cpars(3) = beta / random_gamma( alpha , .TRUE.)

END SUBROUTINE Sigma2_Updater

SUBROUTINE Tau2I_Updater(tau2Iprior, Cpars, MInst)
  
  USE random
  USE TUPDATER_DATA

  IMPLICIT NONE

  DOUBLE PRECISION  ::  tau2Iprior(*), Cpars(*), MInst
  DOUBLE PRECISION  ::  TMatDiff(1:NLOCS, 1:NYEARS)
  REAL              ::  TauAlpha, TauBeta
  !
  !UPDATES the variance of the instrumental error in the main BARCAST code
  !
  ! INVERSE GAMMA
  !
  
  !need to find the sum of the squared residuals between the Inst
  !observations and the corresponding Temp values:
  !extract the Inst part of the T,D matrices:
  !also get rid of the time=0 value from the temp mat:
  !we don't actually need the HH_Select matrix: just take the difference
  !between the Temperature matrix and ther Inst Obs matrix:: wherever there
  !is a NaN in the Data matrix, there will be a NaN in the difference. Then
  !find all non NaN entries, square them, add them up.
  WHERE( DATA_ALL(1:NLOCS, 1:NYEARS) == -99 )
    TMatDiff = 0;
  ELSEWHERE  
    TMatDiff = T_MATRIX(1:NLOCS, 1:NYEARS) - DATA_ALL(1:NLOCS, 1:NYEARS)
  END WHERE
  
  !find the sum of the squares of the residuals::
  
  !we can now calculate the postertior beta parameter:
  TauBeta = SUM(TMatDiff**2)/2.0D0 + tau2Iprior(2)
  !can then calculate the first parameter, alpha, for the posterior inv-gamma
  !dist.:
  TauAlpha = MInst / 2.0D0 + tau2Iprior(1)
  !print(paste("alpha post", alpha.post, "; beta post", beta.post))
  
  !make the draw:
  Cpars(6) = TauBeta/ random_gamma( TauAlpha , .TRUE.)

!  print *, TauBeta
!  print *, TauAlpha
  
END SUBROUTINE Tau2I_Updater


SUBROUTINE Phi_Updater( phiprior, Cpars, inv_scm, DistMat, MHParsLogphi)

  USE random
  USE TUPDATER_DATA

  IMPLICIT INTEGER(I-K)

  DOUBLE PRECISION  ::  phiprior(*), Cpars(*), MHParsLogphi(*)
  DOUBLE PRECISION, DIMENSION(1:NLOCS,1:NYEARS)  ::  TMatDiff, W_now, W_prop
  DOUBLE PRECISION, DIMENSION(1:NLOCS,1:NLOCS)   ::  scm_prop, scm_now, AT_prop, AT_now, DistMat, Eye, inv_scm
  DOUBLE PRECISION  ::  w_squared_now, w_squared_prop, J_par, lgphi_prop
  DOUBLE PRECISION  ::  acceptance, Met_ratio, log_met_ratio, tester, VAE_prop
  DOUBLE PRECISION, ALLOCATABLE :: lgphiVAE(:,:)
  INTEGER  :: INFO, N_Its
  INTEGER, DIMENSION(1:NLOCS, 1:NLOCS)  ::  IEye

  !UPDATES the range parameter in the spatial covaraince matrix for the main
  !BARCAST code
  Eye = 0.0
  DO I = 1, NLOCS
    Eye(I,I) = 1.0
  ENDDO
  IEye = 0
  DO I = 1, NLOCS
    IEye(I,I) = 1
  ENDDO

  J_par = MHparsLogphi(1)**.5;
  N_Its = floor(MHparsLogphi(2) );

  !find the matrix of adjusted temeprature differences that appear in the
  !quadratic forms. 
  TMatDiff = T_MATRIX(1:NLOCS,1:NYEARS)- Cpars(1)*T_MATRIX(1:NLOCS,0:NYEARS-1) - (1-Cpars(1))*Cpars(2)

  !label the spatial correlation matrix and inverse:
  scm_Now = exp(-Cpars(4)*DistMat);
  AT_now = scm_Now
  ! Cholesky decomposition
  !AT_now <- t(chol(scm_Now))
  ! R returns the upper triangle, transpose in next step!
  CALL DPOTRF( 'U', NLOCS, AT_now, NLOCS, INFO)
  
  IF ( INFO .ne. 0) print *, "ERROR!"
  ! Don't need to clear the lower triangular part of AT. The DTRTRS function
  ! references the upper ('U') part only anyways
  !DO I = 2, NLOCS
  !  DO J = 1, I-1
  !    AT_now(I,J) = 0.0
  !  ENDDO
  !ENDDO
  w_now = TMatDiff
  CALL DTRTRS('U', 'T', 'N', NLOCS, NYEARS, AT_now, NLOCS, w_now, NLOCS, INFO )
  IF ( INFO .ne. 0) print *, "ERROR!"
  w_squared_now = (sum(w_now**2))
  !we will also make a "proposed" versions, and replace the "now" versions as
  !necessary. This is an attempt to avoid unnecessary matrix inversions. 

  !Create an empty matrix, N_Its+1 by 2. Left Column: log_phi values, starting
  !with the log of the inputed phi value and ending with the value after N_Its of the
  !Metropolis algorithm. Right column: value of the argument of the
  !exponenent given the value of Beta_1. 

  ALLOCATE(lgphiVAE(1:N_Its+1,2))
  lgphiVAE(1,1) = log(Cpars(4))

  !Find the argument of the exponent for the current logphi value, once more
  !using the matrix then array multiplication trick:
  lgphiVAE(1,2) = -(lgphiVAE(1,1)-phiprior(1))**2/(2*phiprior(2))  - w_squared_now / (2*Cpars(3))
  
  acceptance = 0;
  DO K = 1, N_Its
    lgphi_prop = rnor()
    lgphi_prop = lgphi_prop * J_par + lgphiVAE(K,1) 
    !calculate the spatial correlation matrix using this phi:
    scm_prop = exp(-exp(lgphi_prop)*DistMat);

    AT_prop = scm_prop
    CALL DPOTRF( 'U', NLOCS, AT_prop, NLOCS, INFO)
    IF ( INFO .ne. 0) print *, "ERROR!"

    w_prop = TMatDiff
    CALL DTRTRS('U', 'T', 'N', NLOCS, NYEARS, AT_prop, NLOCS, w_prop, NLOCS, INFO )
    IF ( INFO .ne. 0) print *, "ERROR!"
    w_squared_prop = sum(w_prop**2)

    !calculate the value of the arg of the exponent for this proposed
    !log_phi:
    VAE_prop = -(lgphi_prop - phiprior(1))**2/(2*phiprior(2))  - w_squared_prop / (2*Cpars(3))

    !calculate the log metropolis ratio, using logs to avoid NaNs and other
    !foolishness. 

    log_Met_ratio = VAE_prop - lgphiVAE(K,2) + (NYEARS)* &
      (log(PRODUCT(AT_now, MASK = IEye == 1)) - log(PRODUCT(AT_prop, MASK = IEye == 1)) )
                        ! 2*NYEARS/2 because of AT just being the "root" of R

    Met_ratio = exp(log_Met_ratio);
    !decide on the next value of logphi and arg of exp, and update the values of (i)scm_Now:
    IF (Met_ratio > 1) THEN
      !if ratio greater than 1, accept:
      lgphiVAE(K+1, 1) = lgphi_prop
      lgphiVAE(K+1, 2) = VAE_prop
      scm_Now = scm_prop
      AT_now = AT_prop
      w_now = w_prop
      w_squared_now = w_squared_prop
      acceptance = acceptance + 1
    ELSE
      !else accept with probability equal to the ratio
      tester = uni()
      IF ( Met_ratio > tester) THEN
        lgphiVAE(K+1, 1) = lgphi_prop
        lgphiVAE(K+1, 2) = VAE_prop
        scm_Now = scm_prop
        AT_now = AT_prop
        w_now = w_prop
        w_squared_now = w_squared_prop
        acceptance = acceptance + 1
      ELSE
        lgphiVAE(K + 1,1:2) = lgphiVAE(K, 1:2)
            !no need to update the (i)scm_Now.
      ENDIF
    ENDIF
  ENDDO
  Cpars(4) = EXP(lgphiVAE(N_Its+1 ,1) )
  Cpars(5) = 0
  scm_now = EXP( - Cpars(4)* DistMat)
  Ciscm = Eye
  CALL DPOSV('U', NLOCS, NLOCS, scm_now, NLOCS, Ciscm, NLOCS, Info)
  inv_scm = CISCM
  write (*, '(A, F7.2, E9.2 )'), "acceptance / last step:", acceptance/N_Its, EXP(lgphiVAE(N_Its+1 ,1) )
  DEALLOCATE(lgphiVAE)

END SUBROUTINE Phi_Updater

